import zipfile
from io import BytesIO

import pytest

from octopus.utils import (
    check_file_exists,
    check_is_binary_non_empty,
    check_is_file_valid,
    compute_hash,
    ensure_file_exists,
    extract_zip,
    extract_zip_in_mem_generator,
    get_mime_type,
)


def test_check_file_exists(tmp_path):
    path = tmp_path / "test_file.txt"
    path.touch()
    fp, exist = check_file_exists(str(path))

    assert exist
    assert path == fp


def test_check_file_exists_nonexistent_file(tmp_path):
    path = tmp_path / "test_file.txt"
    fp, exist = check_file_exists(str(path))

    assert not exist
    assert path == fp


def test_check_is_binary_non_empty():
    binary = b"This is a test"
    assert check_is_binary_non_empty(binary)


def test_check_is_binary_non_empty_false():
    binary = b""
    assert not check_is_binary_non_empty(binary)


def test_check_is_file_valid(tmp_path):
    path = tmp_path / "test_file.txt"
    with path.open("w") as f:
        f.write("This is a test")

    assert check_is_file_valid(str(path))


def test_check_is_file_valid_empty_file(tmp_path):
    path = tmp_path / "test_file.txt"
    path.touch()

    assert not check_is_file_valid(str(path))


def test_compute_hash():
    bytes_io = BytesIO(b"This is a test")
    compute_hash(bytes_io)


def test_ensure_file_exists(tmp_path):
    path = tmp_path / "test_file.txt"
    path.touch()
    fp = ensure_file_exists(str(path))

    assert path == fp


def test_ensure_file_exists_nonexistent_file(tmp_path):
    path = tmp_path / "test_file.txt"

    with pytest.raises(FileNotFoundError, match=f"File `{path}` does not exist."):
        ensure_file_exists(str(path))


def test_extract_zip(tmp_path):
    path = tmp_path / "test_file.zip"
    with zipfile.ZipFile(path, "w") as zf:
        zf.writestr("test_file.txt", "This is a test")
        zf.writestr("test_file2.txt", "This is a test2")
        nested_zip = BytesIO()
        with zipfile.ZipFile(nested_zip, "w") as czf:
            czf.writestr("test_file3.txt", "This is a test3")
        nested_zip.seek(0)
        zf.writestr("testzip.zip", nested_zip.read())
    files, invalids = extract_zip(path)

    assert len(files) == 3
    assert len(invalids) == 0


def test_extract_zip_invalid_file(tmp_path):
    path = tmp_path / "test_file.zip"
    with zipfile.ZipFile(path, "w") as zf:
        zf.writestr("test_file.txt", "This is a test")
        zf.writestr("test_file2.txt", "")
    files, invalids = extract_zip(path)

    assert len(files) == 1
    assert len(invalids) == 1


def test_extract_zip_ignore_dir_ignore_dot(tmp_path):
    path = tmp_path / "test_file.zip"
    with zipfile.ZipFile(path, "w") as zf:
        zf.writestr(".test_dir/test_file.txt", "This is a test")
        zf.writestr("test_dir/test_file.txt", "This is a test")
        zf.writestr("test_dir/.test_file.txt", "This is a test")
    files, invalids = extract_zip(path)

    assert len(files) == 1
    assert len(invalids) == 0


def test_extract_zip_in_mem_generator():
    zbio = BytesIO()
    with zipfile.ZipFile(zbio, "w") as zf:
        zf.writestr("test_file.txt", "This is a test")
        zf.writestr("test_file2.txt", "This is a test2")
        nested_zip = BytesIO()
        with zipfile.ZipFile(nested_zip, "w") as czf:
            czf.writestr("test_file3.txt", "This is a test3")
        nested_zip.seek(0)
        zf.writestr("testzip.zip", nested_zip.read())
    zbio.seek(0)
    files, invalids = [], []
    for content, fname in extract_zip_in_mem_generator(zbio):
        print(fname)
        if content:
            files.append(content)
        else:
            invalids.append(content)

    assert len(files) == 3
    assert len(invalids) == 0


def test_extract_zip_in_mem_generator_invalid_file():
    zbio = BytesIO()
    with zipfile.ZipFile(zbio, "w") as zf:
        zf.writestr("test_file.txt", "This is a test")
        zf.writestr("test_file2.txt", "")
    zbio.seek(0)
    files, invalids = [], []
    for content, fname in extract_zip_in_mem_generator(zbio):
        if content:
            files.append(content)
        else:
            invalids.append(content)

    assert len(files) == 1
    assert len(invalids) == 1


def test_extract_zip_in_mem_generator_ignore_dir_ignore_dot():
    zbio = BytesIO()
    with zipfile.ZipFile(zbio, "w") as zf:
        zf.writestr(".test_dir/test_file.txt", "This is a test")
        zf.writestr("test_dir/test_file.txt", "This is a test")
        zf.writestr("test_dir/.test_file.txt", "This is a test")
    zbio.seek(0)
    files, invalids = [], []
    for content, fname in extract_zip_in_mem_generator(zbio):
        if content:
            files.append(content)
        else:
            invalids.append(content)

    assert len(files) == 1
    assert len(invalids) == 0


def test_get_mime_type(tmp_path):
    path = tmp_path / "test_file.txt"
    with path.open("w") as f:
        f.write("This is a test")

    assert get_mime_type(str(path)) == "text/plain"
