import pytest

from octopus.text.date_extraction import _get_today_date, date_extraction


@pytest.fixture(
    params=[
        ##############################
        # Day Month Year
        ##############################
        # 01 Feb 2024
        ("1 Feb 2024", "2024-02-01"),
        ("1Feb2024", "2024-02-01"),
        ("01 Feb 2024", "2024-02-01"),
        ("01Feb2024", "2024-02-01"),
        ("1st Feb 2024", "2024-02-01"),
        ("1stFeb2024", "2024-02-01"),
        ("01st Feb 2024", "2024-02-01"),
        ("01stFeb2024", "2024-02-01"),
        ("2nd Feb 2024", "2024-02-02"),
        ("02nd Feb 2024", "2024-02-02"),
        ("3rd Feb 2024", "2024-02-03"),
        ("03rd Feb 2024", "2024-02-03"),
        ("31 Feb 2024", _get_today_date()),
        # 01 February 2024
        ("01 February 2024", "2024-02-01"),
        ("01February2024", "2024-02-01"),
        ("1st February 2024", "2024-02-01"),
        ("1stFebruary2024", "2024-02-01"),
        ("01st February 2024", "2024-02-01"),
        ("01stFebruary2024", "2024-02-01"),
        ("2nd February 2024", "2024-02-02"),
        ("02nd February 2024", "2024-02-02"),
        ("3rd February 2024", "2024-02-03"),
        ("03rd February 2024", "2024-02-03"),
        ("31 February 2024", _get_today_date()),
        ("31February2024", _get_today_date()),
        # 01-02-2024
        ("25-12-1981", "1981-12-25"),
        ("31-02-2024", _get_today_date()),
        # 01-Feb-2024
        ("01-Feb-2024", "2024-02-01"),
        ("31-Feb-2024", _get_today_date()),
        # 01-February-2024
        ("01-February-2024", "2024-02-01"),
        ("31-February-2024", _get_today_date()),
        # 01/02/2024
        ("01/02/2024", "2024-02-01"),
        ("31/02/2024", _get_today_date()),
        ##############################
        # Month Day Year
        ##############################
        # Feb 01 2024
        ("Feb 01 2024", "2024-02-01"),
        ("Feb01 2024", "2024-02-01"),
        ("Feb 1st 2024", "2024-02-01"),
        ("Feb1st 2024", "2024-02-01"),
        ("Feb 01st 2024", "2024-02-01"),
        ("Feb01st 2024", "2024-02-01"),
        ("Feb 2nd 2024", "2024-02-02"),
        ("Feb 02nd 2024", "2024-02-02"),
        ("Feb 3rd 2024", "2024-02-03"),
        ("Feb 03rd 2024", "2024-02-03"),
        ("Feb 31 2024", _get_today_date()),
        ("Feb31 2024", _get_today_date()),
        # February 01 2024
        ("February 01 2024", "2024-02-01"),
        ("February01 2024", "2024-02-01"),
        ("February 1st 2024", "2024-02-01"),
        ("February1st 2024", "2024-02-01"),
        ("February 01st 2024", "2024-02-01"),
        ("February01st 2024", "2024-02-01"),
        ("February 2nd 2024", "2024-02-02"),
        ("February 02nd 2024", "2024-02-02"),
        ("February 3rd 2024", "2024-02-03"),
        ("February 03rd 2024", "2024-02-03"),
        ("February 31 2024", _get_today_date()),
        ("February31 2024", _get_today_date()),
        # Feb 01, 2024
        ("Feb 01, 2024", "2024-02-01"),
        ("Feb01, 2024", "2024-02-01"),
        ("Feb 1st, 2024", "2024-02-01"),
        ("Feb1st, 2024", "2024-02-01"),
        ("Feb 01st, 2024", "2024-02-01"),
        ("Feb01st, 2024", "2024-02-01"),
        ("Feb 2nd, 2024", "2024-02-02"),
        ("Feb 02nd, 2024", "2024-02-02"),
        ("Feb 3rd, 2024", "2024-02-03"),
        ("Feb 03rd, 2024", "2024-02-03"),
        ("Feb 31, 2024", _get_today_date()),
        ("Feb31, 2024", _get_today_date()),
        # February 01, 2024
        ("February 01, 2024", "2024-02-01"),
        ("February01, 2024", "2024-02-01"),
        ("February 1st, 2024", "2024-02-01"),
        ("February1st, 2024", "2024-02-01"),
        ("February 01st, 2024", "2024-02-01"),
        ("February01st, 2024", "2024-02-01"),
        ("February 2nd, 2024", "2024-02-02"),
        ("February 02nd, 2024", "2024-02-02"),
        ("February 3rd, 2024", "2024-02-03"),
        ("February 03rd, 2024", "2024-02-03"),
        ("February 31, 2024", _get_today_date()),
        ("February31, 2024", _get_today_date()),
        ##############################
        # Year Month Day
        ##############################
        # 2024-02-01
        ("2024-02-01", "2024-02-01"),
        ("2024-02-31", _get_today_date()),
        # 2024-Feb-01
        ("2024-Feb-01", "2024-02-01"),
        ("2024-Feb-31", _get_today_date()),
        # 2024-February-01
        ("2024-February-01", "2024-02-01"),
        ("2024-February-31", _get_today_date()),
        # 2024/02/01
        ("2024/02/01", "2024-02-01"),
        ("2024/02/31", _get_today_date()),
        # 2024/Feb/01
        ("2024/Feb/01", "2024-02-01"),
        ("2024/Feb/31", _get_today_date()),
        # 2024/February/01
        ("2024/February/01", "2024-02-01"),
        ("2024/February/31", _get_today_date()),
        ##############################
        # Texts
        ##############################
        ("2 - 1 Dec 20221", _get_today_date()),
        (
            "<p/>\n<p>Credit Review (APPROVED)\nSUBMISSION  : 12/05/2021\n</p>",
            "2021-05-12",
        ),
        ("</p><p>NEXTREVIEWDATE</p><p> 30/06/2022 < /p><p>ajksdjask", "2022-06-30"),
        (
            "</p><p>assignurances</p><p> 29july2019 </p><p/></div></body></html>",
            "2019-07-29",
        ),
    ]
)
def date_extraction_cases(request):
    return request.param


def test_date_extraction(date_extraction_cases):
    txt, expected = date_extraction_cases
    assert expected == date_extraction(txt)
