import json

import pytest

from octopus.clients.wfi_client import WFI_FIELDS
from octopus.data.company_data import tag_company_data, tag_document_type_data


# ==================== TAG COMPANY DATA ==================== #
def test_get_company_data_with_no_company(item):
    tag_company_data(item, {})

    assert item["keywords"] == {
        label: value["default"]
        for label, value in WFI_FIELDS.items()
        if "company" in label
    }


def test_get_company_data_with_one_company(item, company):
    item.update(
        {
            "keywords": {
                "company_cif": ["123456789"],
                "company_name": ["Company A"],
                "company_uid": ["SG_123456789"],
            }
        }
    )
    tag_company_data(item, company)

    assert item["keywords"] == {
        "company_cif": ["123456789"],
        "company_name": ["Company A"],
        "company_uid": ["SG_123456789"],
        "wfi_company_name": ["Company A"],
        "wfi_company_cif": ["123456789"],
        "wfi_company_rm_code": ["R012"],
        "wfi_company_segment": ["R"],
        "wfi_company_team_name": ["Team 1"],
    }


def test_get_company_data_with_multiple_companies(item, company) -> None:
    item.update(
        {
            "keywords": {
                "company_cif": ["123456789", "123456790"],
                "company_name": ["Company A", "Company B"],
                "company_uid": ["SG_123456789", "SG_123456790"],
            }
        }
    )
    tag_company_data(item, company)

    assert item["keywords"] == {
        "company_cif": ["123456789", "123456790"],
        "company_name": ["Company A", "Company B"],
        "company_uid": ["SG_123456789", "SG_123456790"],
        "wfi_company_cif": ["123456789"],
        "wfi_company_name": ["Company A"],
        "wfi_company_rm_code": ["R012"],
        "wfi_company_segment": ["R"],
        "wfi_company_team_name": ["Team 1"],
    }


# ==================== TAG DOCUMENT TYPE DATA ==================== #
def test_get_document_type_data_unclassified(item, mock_redis_client):
    mock_redis_client.hget.return_value = None
    tag_document_type_data(item, mock_redis_client)

    assert item["keywords"] == {
        label: value["default"]
        for label, value in WFI_FIELDS.items()
        if label in ("wfi_document_type", "wfi_document_category", "wfi_document_name")
    }


def test_get_document_type_data(item, mock_redis_client):
    sample_mapping = {
        "document_category": "Document Category",
        "wfi_document_type": "WFI Document Type",
        "wfi_document_category": "WFI Document Category",
        "wfi_document_name": "WFI Document Name",
    }

    mock_redis_client.hget.return_value = json.dumps(sample_mapping).encode()
    tag_document_type_data(item, mock_redis_client)

    assert item["keywords"] == {key: [value] for key, value in sample_mapping.items()}


@pytest.fixture
def company():
    return {
        "company_cif": ["123456789"],
        "company_name": ["Company A"],
        "company_uid": ["SG_123456789"],
        "wfi_company_rm_code": ["R012"],
        "wfi_company_segment": ["R"],
        "wfi_company_team_code": ["T002"],
        "wfi_company_team_name": ["Team 1"],
    }


@pytest.fixture
def item():
    return {
        "id": "item-id-1",
        "title": "Test Item",
        "keywords": {},
    }
