import re

import pytest

from octopus.data.company_data.company_data_index import (
    CompanyDataIndex,
    _generate_regexes_lookup,
)

TEST_CASES = [
    ("Test AS", "test as", {"test as": "Test AS"}),
    ("Test A.S", "test a.s", {"test a.s": "Test A.S"}),
    (
        "Test A.S.",
        r"test(?: aksjeselskap| a\. s\.| a\.s\.)",
        {
            "test aksjeselskap": "Test A.S.",
            "test a. s.": "Test A.S.",
            "test a.s.": "Test A.S.",
        },
    ),
    (
        "Test A. S.",
        r"test(?: aksjeselskap| a\. s\.| a\.s\.)",
        {
            "test aksjeselskap": "Test A. S.",
            "test a. s.": "Test A. S.",
            "test a.s.": "Test A. S.",
        },
    ),
    (
        "Test Aksjeselskap",
        r"test(?: aksjeselskap| a\. s\.| a\.s\.)",
        {
            "test aksjeselskap": "Test Aksjeselskap",
            "test a. s.": "Test Aksjeselskap",
            "test a.s.": "Test Aksjeselskap",
        },
    ),
    ("Test AB", "test ab", {"test ab": "Test AB"}),
    ("Test A.B", "test a.b", {"test a.b": "Test A.B"}),
    (
        "Test A.B.",
        r"test(?: aktiebolag| a\. b\.| a\.b\.)",
        {
            "test aktiebolag": "Test A.B.",
            "test a. b.": "Test A.B.",
            "test a.b.": "Test A.B.",
        },
    ),
    (
        "Test A. B. ",
        r"test(?: aktiebolag| a\. b\.| a\.b\.)",
        {
            "test aktiebolag": "Test A. B. ",
            "test a. b.": "Test A. B. ",
            "test a.b.": "Test A. B. ",
        },
    ),
    (
        "Test Aktiebolag",
        r"test(?: aktiebolag| a\. b\.| a\.b\.)",
        {
            "test aktiebolag": "Test Aktiebolag",
            "test a. b.": "Test Aktiebolag",
            "test a.b.": "Test Aktiebolag",
        },
    ),
    ("Test AG", "test ag", {"test ag": "Test AG"}),
    ("Test A.G", "test a.g", {"test a.g": "Test A.G"}),
    (
        "Test A.G.",
        r"test(?: aktiengesellschaft| a\. g\.| a\.g\.)",
        {
            "test aktiengesellschaft": "Test A.G.",
            "test a. g.": "Test A.G.",
            "test a.g.": "Test A.G.",
        },
    ),
    (
        "Test A. G.",
        r"test(?: aktiengesellschaft| a\. g\.| a\.g\.)",
        {
            "test aktiengesellschaft": "Test A. G.",
            "test a. g.": "Test A. G.",
            "test a.g.": "Test A. G.",
        },
    ),
    (
        "Test Aktiengesellschaft",
        r"test(?: aktiengesellschaft| a\. g\.| a\.g\.)",
        {
            "test aktiengesellschaft": "Test Aktiengesellschaft",
            "test a. g.": "Test Aktiengesellschaft",
            "test a.g.": "Test Aktiengesellschaft",
        },
    ),
    ("Test BV", "test bv", {"test bv": "Test BV"}),
    ("Test B.V", "test b.v", {"test b.v": "Test B.V"}),
    (
        "Test B.V.",
        r"test(?: besloten vennootschap| b\. v\.| b\.v\.)",
        {
            "test besloten vennootschap": "Test B.V.",
            "test b. v.": "Test B.V.",
            "test b.v.": "Test B.V.",
        },
    ),
    (
        "Test B. V.",
        r"test(?: besloten vennootschap| b\. v\.| b\.v\.)",
        {
            "test besloten vennootschap": "Test B. V.",
            "test b. v.": "Test B. V.",
            "test b.v.": "Test B. V.",
        },
    ),
    (
        "Test Besloten vennootschap",
        r"test(?: besloten vennootschap| b\. v\.| b\.v\.)",
        {
            "test besloten vennootschap": "Test Besloten vennootschap",
            "test b. v.": "Test Besloten vennootschap",
            "test b.v.": "Test Besloten vennootschap",
        },
    ),
    ("Test Coop", "test coop", {"test coop": "Test Coop"}),
    ("Test CoOp", "test coop", {"test coop": "Test CoOp"}),
    (
        "Test Co-op",
        "test(?: cooperative| co-op)",
        {"test cooperative": "Test Co-op", "test co-op": "Test Co-op"},
    ),
    (
        "Test Co-Op",
        "test(?: cooperative| co-op)",
        {"test cooperative": "Test Co-Op", "test co-op": "Test Co-Op"},
    ),
    (
        "Test Cooperative",
        "test(?: cooperative| co-op)",
        {"test cooperative": "Test Cooperative", "test co-op": "Test Cooperative"},
    ),
    ("Test Incorp", "test incorp", {"test incorp": "Test Incorp"}),
    (
        "Test Incorporation",
        "test(?: incorporation)",
        {"test incorporation": "Test Incorporation"},
    ),
    ("Test Corp", "test corp", {"test corp": "Test Corp"}),
    (
        "Test Corp.",
        r"test(?: corporation| corp\.)",
        {"test corporation": "Test Corp.", "test corp.": "Test Corp."},
    ),
    (
        "Test Corporation",
        r"test(?: corporation| corp\.)",
        {"test corporation": "Test Corporation", "test corp.": "Test Corporation"},
    ),
    (
        "Wintrust Financial Corporation",
        r"wintrust financial(?: corporation| corp\.)",
        {
            "wintrust financial corporation": "Wintrust Financial Corporation",
            "wintrust financial corp.": "Wintrust Financial Corporation",
        },
    ),
    (
        "Westamerica Bancorporation",
        r"westamerica bancorporation",
        {"westamerica bancorporation": "Westamerica Bancorporation"},
    ),
    (
        "Test Gesellschaft mit beschrankter Haftung",
        (
            r"test(?: gesellschaft mit beschrankter haftung| "
            r"gesellschaft mit beschrnkter haftung| gmbh)"
        ),
        {
            "test gesellschaft mit beschrankter haftung": (
                "Test Gesellschaft mit beschrankter Haftung"
            ),
            "test gesellschaft mit beschrnkter haftung": (
                "Test Gesellschaft mit beschrankter Haftung"
            ),
            "test gmbh": "Test Gesellschaft mit beschrankter Haftung",
        },
    ),
    (
        "Test GmbH",
        (
            r"test(?: gesellschaft mit beschrankter haftung| "
            r"gesellschaft mit beschrnkter haftung| gmbh)"
        ),
        {
            "test gesellschaft mit beschrankter haftung": "Test GmbH",
            "test gesellschaft mit beschrnkter haftung": "Test GmbH",
            "test gmbh": "Test GmbH",
        },
    ),
    ("Test Inc", "test inc", {"test inc": "Test Inc"}),
    (
        "Test Incorporated",
        r"test(?: incorporated| inc\.)",
        {"test incorporated": "Test Incorporated", "test inc.": "Test Incorporated"},
    ),
    (
        "Washington Federal, Inc.",
        r"washington federal(?: incorporated| inc\.)",
        {
            "washington federal incorporated": "Washington Federal, Inc.",
            "washington federal inc.": "Washington Federal, Inc.",
        },
    ),
    (
        "Washington Trust Bancorp, Inc.",
        r"washington trust bancorp(?: incorporated| inc\.)",
        {
            "washington trust bancorp incorporated": "Washington Trust Bancorp, Inc.",
            "washington trust bancorp inc.": "Washington Trust Bancorp, Inc.",
        },
    ),
    (
        "Wins Finance Holdings Inc.",
        r"wins finance holdings(?: incorporated| inc\.)",
        {
            "wins finance holdings incorporated": "Wins Finance Holdings Inc.",
            "wins finance holdings inc.": "Wins Finance Holdings Inc.",
        },
    ),
    (
        "Viacom Inc.",
        r"viacom(?: incorporated| inc\.)",
        {"viacom incorporated": "Viacom Inc.", "viacom inc.": "Viacom Inc."},
    ),
    (
        "Village Super Market, Inc.",
        r"village super market(?: incorporated| inc\.)",
        {
            "village super market incorporated": "Village Super Market, Inc.",
            "village super market inc.": "Village Super Market, Inc.",
        },
    ),
    (
        "Westwater Resources, Inc.",
        r"westwater resources(?: incorporated| inc\.)",
        {
            "westwater resources incorporated": "Westwater Resources, Inc.",
            "westwater resources inc.": "Westwater Resources, Inc.",
        },
    ),
    (
        "Virtus Investment Partners, Inc.",
        r"virtus investment partners(?: incorporated| inc\.)",
        {
            "virtus investment partners incorporated": "Virtus Investment Partners, Inc.",
            "virtus investment partners inc.": "Virtus Investment Partners, Inc.",
        },
    ),
    (
        "Wheeler Real Estate Investment Trust, Inc.",
        r"wheeler real estate investment trust(?: incorporated| inc\.)",
        {
            "wheeler real estate investment trust incorporated": (
                "Wheeler Real Estate Investment Trust, Inc."
            ),
            "wheeler real estate investment trust inc.": (
                "Wheeler Real Estate Investment Trust, Inc."
            ),
        },
    ),
    (
        "Vertex Pharmaceuticals Incorporated",
        r"vertex pharmaceuticals(?: incorporated| inc\.)",
        {
            "vertex pharmaceuticals incorporated": "Vertex Pharmaceuticals Incorporated",
            "vertex pharmaceuticals inc.": "Vertex Pharmaceuticals Incorporated",
        },
    ),
    (
        "ViaSat, Inc.",
        r"viasat(?: incorporated| inc\.)",
        {"viasat incorporated": "ViaSat, Inc.", "viasat inc.": "ViaSat, Inc."},
    ),
    (
        "ViaSat, Inc. Lol Inc.",
        r"viasat inc. lol(?: incorporated| inc\.)",
        {
            "viasat inc. lol incorporated": "ViaSat, Inc. Lol Inc.",
            "viasat inc. lol inc.": "ViaSat, Inc. Lol Inc.",
        },
    ),
    (
        "ViaSat, Inc. Incorporated Inc.",
        r"viasat inc. incorporated(?: incorporated| inc\.)",
        {
            "viasat inc. incorporated incorporated": "ViaSat, Inc. Incorporated Inc.",
            "viasat inc. incorporated inc.": "ViaSat, Inc. Incorporated Inc.",
        },
    ),
    (
        "Test Kommanditgesellschaft auf Aktien",
        r"test(?: kommanditgesellschaft auf aktien| k\. ga\. a\.| k\.ga\.a\.| kgaa)",
        {
            "test kommanditgesellschaft auf aktien": (
                "Test Kommanditgesellschaft auf Aktien"
            ),
            "test k. ga. a.": "Test Kommanditgesellschaft auf Aktien",
            "test k.ga.a.": "Test Kommanditgesellschaft auf Aktien",
            "test kgaa": "Test Kommanditgesellschaft auf Aktien",
        },
    ),
    (
        "Test KGaA",
        r"test(?: kommanditgesellschaft auf aktien| k\. ga\. a\.| k\.ga\.a\.| kgaa)",
        {
            "test kommanditgesellschaft auf aktien": "Test KGaA",
            "test k. ga. a.": "Test KGaA",
            "test k.ga.a.": "Test KGaA",
            "test kgaa": "Test KGaA",
        },
    ),
    (
        "Test Limited Liability Company",
        r"test(?: limited liability company| l\. l\. c\.| l\.l\.c\.| llc)",
        {
            "test limited liability company": "Test Limited Liability Company",
            "test l. l. c.": "Test Limited Liability Company",
            "test l.l.c.": "Test Limited Liability Company",
            "test llc": "Test Limited Liability Company",
        },
    ),
    (
        "Test LLC",
        r"test(?: limited liability company| l\. l\. c\.| l\.l\.c\.| llc)",
        {
            "test limited liability company": "Test LLC",
            "test l. l. c.": "Test LLC",
            "test l.l.c.": "Test LLC",
            "test llc": "Test LLC",
        },
    ),
    ("Test Co", "test co", {"test co": "Test Co"}),
    (
        "ViaSat Company",
        r"viasat(?: company| co\.)",
        {
            "viasat company": "ViaSat Company",
            "viasat co.": "ViaSat Company",
        },
    ),
    (
        "Test Company",
        r"test(?: company| co\.)",
        {"test company": "Test Company", "test co.": "Test Company"},
    ),
    (
        "Test LLP",
        r"test(?: limited liability partnership| l\. l\. p\.| l\.l\.p\.| llp)",
        {
            "test limited liability partnership": "Test LLP",
            "test l. l. p.": "Test LLP",
            "test l.l.p.": "Test LLP",
            "test llp": "Test LLP",
        },
    ),
    (
        "Test Limited Liability Partnership",
        r"test(?: limited liability partnership| l\. l\. p\.| l\.l\.p\.| llp)",
        {
            "test limited liability partnership": "Test Limited Liability Partnership",
            "test l. l. p.": "Test Limited Liability Partnership",
            "test l.l.p.": "Test Limited Liability Partnership",
            "test llp": "Test Limited Liability Partnership",
        },
    ),
    ("Test LP", "test lp", {"test lp": "Test LP"}),
    (
        "Test Limited Partnership",
        r"test(?: limited partnership| l\. p\.| l\.p\.)",
        {
            "test limited partnership": "Test Limited Partnership",
            "test l. p.": "Test Limited Partnership",
            "test l.p.": "Test Limited Partnership",
        },
    ),
    ("Test NV", "test nv", {"test nv": "Test NV"}),
    (
        "Test Naamloze Vennootschap",
        r"test(?: naamloze vennootschap| n\. v\.| n\.v\.)",
        {
            "test naamloze vennootschap": "Test Naamloze Vennootschap",
            "test n. v.": "Test Naamloze Vennootschap",
            "test n.v.": "Test Naamloze Vennootschap",
        },
    ),
    (
        "Test Osakeyhtio",
        r"test(?: osakeyhtio| osakeyhti| oy\.)",
        {
            "test osakeyhtio": "Test Osakeyhtio",
            "test osakeyhti": "Test Osakeyhtio",
            "test oy.": "Test Osakeyhtio",
        },
    ),
    ("Test Oy", "test oy", {"test oy": "Test Oy"}),
    (
        "Test Pte. Ltd.",
        r"test(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "test private limited": "Test Pte. Ltd.",
            "test pte. ltd.": "Test Pte. Ltd.",
            "test pte.ltd.": "Test Pte. Ltd.",
            "test pte ltd": "Test Pte. Ltd.",
        },
    ),
    (
        "Test Pte.Ltd.",
        r"test(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "test private limited": "Test Pte.Ltd.",
            "test pte. ltd.": "Test Pte.Ltd.",
            "test pte.ltd.": "Test Pte.Ltd.",
            "test pte ltd": "Test Pte.Ltd.",
        },
    ),
    (
        "Test Pte.Ltd",
        r"test pte.ltd",
        {
            "test pte.ltd": "Test Pte.Ltd",
        },
    ),
    (
        "Test Private Limited",
        r"test(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "test private limited": "Test Private Limited",
            "test pte. ltd.": "Test Private Limited",
            "test pte.ltd.": "Test Private Limited",
            "test pte ltd": "Test Private Limited",
        },
    ),
    (
        "Test Pte Ltd",
        r"test(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "test private limited": "Test Pte Ltd",
            "test pte. ltd.": "Test Pte Ltd",
            "test pte.ltd.": "Test Pte Ltd",
            "test pte ltd": "Test Pte Ltd",
        },
    ),
    (
        "Test Limited",
        "test(?: limited| ltd)",
        {
            "test limited": "Test Limited",
            "test ltd": "Test Limited",
        },
    ),
    (
        "Test Ltd",
        "test(?: limited| ltd)",
        {
            "test limited": "Test Ltd",
            "test ltd": "Test Ltd",
        },
    ),
    ("Test Pty", "test pty", {"test pty": "Test Pty"}),
    (
        "Test Proprietary",
        r"test(?: proprietary| pty\.)",
        {"test proprietary": "Test Proprietary", "test pty.": "Test Proprietary"},
    ),
    (
        "Test Pty.",
        r"test(?: proprietary| pty\.)",
        {
            "test proprietary": "Test Pty.",
            "test pty.": "Test Pty.",
        },
    ),
    (
        "Test Sendirian Berhad",
        r"test(?: sendirian berhad| sdn\. bhd\.| sdn\.bhd\.| sdn bhd)",
        {
            "test sendirian berhad": "Test Sendirian Berhad",
            "test sdn. bhd.": "Test Sendirian Berhad",
            "test sdn.bhd.": "Test Sendirian Berhad",
            "test sdn bhd": "Test Sendirian Berhad",
        },
    ),
    (
        "Test Sdn Bhd",
        r"test(?: sendirian berhad| sdn\. bhd\.| sdn\.bhd\.| sdn bhd)",
        {
            "test sendirian berhad": "Test Sdn Bhd",
            "test sdn. bhd.": "Test Sdn Bhd",
            "test sdn.bhd.": "Test Sdn Bhd",
            "test sdn bhd": "Test Sdn Bhd",
        },
    ),
    (
        "Test Berhad",
        "test(?: berhad| bhd)",
        {"test berhad": "Test Berhad", "test bhd": "Test Berhad"},
    ),
    (
        "Test Bhd",
        "test(?: berhad| bhd)",
        {"test berhad": "Test Bhd", "test bhd": "Test Bhd"},
    ),
    ("Test SA", "test sa", {"test sa": "Test SA"}),
    ("Test S.A", "test s.a", {"test s.a": "Test S.A"}),
    (
        "Test S.A.",
        r"test(?: societe anonyme| socit anonyme| s\. a\.| s\.a\.)",
        {
            "test societe anonyme": "Test S.A.",
            "test socit anonyme": "Test S.A.",
            "test s. a.": "Test S.A.",
            "test s.a.": "Test S.A.",
        },
    ),
    (
        "Test S. A.",
        r"test(?: societe anonyme| socit anonyme| s\. a\.| s\.a\.)",
        {
            "test societe anonyme": "Test S. A.",
            "test socit anonyme": "Test S. A.",
            "test s. a.": "Test S. A.",
            "test s.a.": "Test S. A.",
        },
    ),
    (
        "Test Soci t  Anonyme",
        "test soci t anonyme",
        {"test soci t anonyme": "Test Soci t  Anonyme"},
    ),
    (
        "Test Socit  Anonyme",
        r"test(?: societe anonyme| socit anonyme| s\. a\.| s\.a\.)",
        {
            "test societe anonyme": "Test Socit  Anonyme",
            "test socit anonyme": "Test Socit  Anonyme",
            "test s. a.": "Test Socit  Anonyme",
            "test s.a.": "Test Socit  Anonyme",
        },
    ),
    (
        "Test Societe Anonyme",
        r"test(?: societe anonyme| socit anonyme| s\. a\.| s\.a\.)",
        {
            "test societe anonyme": "Test Societe Anonyme",
            "test socit anonyme": "Test Societe Anonyme",
            "test s. a.": "Test Societe Anonyme",
            "test s.a.": "Test Societe Anonyme",
        },
    ),
    ("Test SE", "test se", {"test se": "Test SE"}),
    (
        "Test Societas Europaea",
        r"test(?: societas europaea| s\. e\.| s\.e\.)",
        {
            "test societas europaea": "Test Societas Europaea",
            "test s. e.": "Test Societas Europaea",
            "test s.e.": "Test Societas Europaea",
        },
    ),
    (
        "VictoryShares International High Div Volatility Wtd ETF",
        "victoryshares international high div volatility wtd etf",
        {
            "victoryshares international high div volatility wtd etf": (
                "VictoryShares International High Div Volatility Wtd ETF"
            )
        },
    ),
    (
        "VictoryShares US Small Cap Volatility Wtd ETF",
        "victoryshares us small cap volatility wtd etf",
        {
            "victoryshares us small cap volatility wtd etf": (
                "VictoryShares US Small Cap Volatility Wtd ETF"
            )
        },
    ),
    (
        "Vident International Equity Fund",
        "vident international equity fund",
        {"vident international equity fund": "Vident International Equity Fund"},
    ),
    (
        "Coca-Cola Bottling Co. Consolidated",
        "cocacola bottling co. consolidated",
        {
            "cocacola bottling co. consolidated": "Coca-Cola Bottling Co. Consolidated",
        },
    ),
    (
        "Test Tornado Holdings, Inc.",
        r"test tornado holdings(?: incorporated| inc\.)",
        {
            "test tornado holdings incorporated": "Test Tornado Holdings, Inc.",
            "test tornado holdings inc.": "Test Tornado Holdings, Inc.",
        },
    ),
    (
        "Lame Joke Broadcast Group, Inc.",
        r"lame joke broadcast group(?: incorporated| inc\.)",
        {
            "lame joke broadcast group incorporated": "Lame Joke Broadcast Group, Inc.",
            "lame joke broadcast group inc.": "Lame Joke Broadcast Group, Inc.",
        },
    ),
    (
        "Iseeyou Corporation",
        r"iseeyou(?: corporation| corp\.)",
        {
            "iseeyou corporation": "Iseeyou Corporation",
            "iseeyou corp.": "Iseeyou Corporation",
        },
    ),
    (
        "QOWFIWQF Holding N.V.",
        r"qowfiwqf holding(?: naamloze vennootschap| n\. v\.| n\.v\.)",
        {
            "qowfiwqf holding naamloze vennootschap": "QOWFIWQF Holding N.V.",
            "qowfiwqf holding n. v.": "QOWFIWQF Holding N.V.",
            "qowfiwqf holding n.v.": "QOWFIWQF Holding N.V.",
        },
    ),
    (
        "Food Value Index ETF",
        "food value index etf",
        {"food value index etf": "Food Value Index ETF"},
    ),
    ("Iwefwef NV", "iwefwef nv", {"iwefwef nv": "Iwefwef NV"}),
    ("What is this ETF", "what is this etf", {"what is this etf": "What is this ETF"}),
    (
        "Squirro Therapeutics plc",
        "squirro therapeutics plc",
        {"squirro therapeutics plc": "Squirro Therapeutics plc"},
    ),
    (
        "US Equity Dividend Select ETF",
        "us equity dividend select etf",
        {"us equity dividend select etf": "US Equity Dividend Select ETF"},
    ),
    (
        "iSectors Post-MPT Growth ETF",
        "isectors postmpt growth etf",
        {"isectors postmpt growth etf": "iSectors Post-MPT Growth ETF"},
    ),
    ("Seefi", "seefi", {"seefi": "Seefi"}),
    (
        "Senior Housing Properties Trust",
        "senior housing properties trust",
        {"senior housing properties trust": "Senior Housing Properties Trust"},
    ),
    (
        "ABCoy Ry Limited",
        r"abcoy ry(?: limited| ltd)",
        {"abcoy ry limited": "ABCoy Ry Limited", "abcoy ry ltd": "ABCoy Ry Limited"},
    ),
    (
        "ABCoy Holdings Limited",
        r"abcoy holdings(?: limited| ltd)",
        {
            "abcoy holdings limited": "ABCoy Holdings Limited",
            "abcoy holdings ltd": "ABCoy Holdings Limited",
        },
    ),
    (
        "Aoyi Capital Management",
        "aoyi capital management",
        {"aoyi capital management": "Aoyi Capital Management"},
    ),
    (
        "Aoyi Capital Management (Singapore) PTE LTD",
        r"aoyi capital management singapore(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "aoyi capital management singapore private limited": (
                "Aoyi Capital Management (Singapore) PTE LTD"
            ),
            "aoyi capital management singapore pte. ltd.": (
                "Aoyi Capital Management (Singapore) PTE LTD"
            ),
            "aoyi capital management singapore pte.ltd.": (
                "Aoyi Capital Management (Singapore) PTE LTD"
            ),
            "aoyi capital management singapore pte ltd": (
                "Aoyi Capital Management (Singapore) PTE LTD"
            ),
        },
    ),
    (
        "Corporation ABC Ltd",
        r"corporation abc(?: limited| ltd)",
        {
            "corporation abc limited": "Corporation ABC Ltd",
            "corporation abc ltd": "Corporation ABC Ltd",
        },
    ),
    (
        "Abcco Finance Group Limited",
        r"abcco finance group(?: limited| ltd)",
        {
            "abcco finance group limited": "Abcco Finance Group Limited",
            "abcco finance group ltd": "Abcco Finance Group Limited",
        },
    ),
    (
        "Abcco SG (xxx) Investments No1 Limited",
        r"abcco sg xxx investments no1(?: limited| ltd)",
        {
            "abcco sg xxx investments no1 limited": "Abcco SG (xxx) Investments No1 Limited",
            "abcco sg xxx investments no1 ltd": "Abcco SG (xxx) Investments No1 Limited",
        },
    ),
    (
        "Abcco Singapore Holding Limited",
        r"abcco singapore holding(?: limited| ltd)",
        {
            "abcco singapore holding limited": "Abcco Singapore Holding Limited",
            "abcco singapore holding ltd": "Abcco Singapore Holding Limited",
        },
    ),
    (
        "Abcco Original Pte Ltd",
        r"abcco original(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "abcco original private limited": "Abcco Original Pte Ltd",
            "abcco original pte. ltd.": "Abcco Original Pte Ltd",
            "abcco original pte.ltd.": "Abcco Original Pte Ltd",
            "abcco original pte ltd": "Abcco Original Pte Ltd",
        },
    ),
    (
        "Abcco Finance Group Limited",
        r"abcco finance group(?: limited| ltd)",
        {
            "abcco finance group limited": "Abcco Finance Group Limited",
            "abcco finance group ltd": "Abcco Finance Group Limited",
        },
    ),
    (
        "SQ & CO. PTE LTD",
        r"sq co.(?: private limited| pte\. ltd\.| pte\.ltd\.| pte ltd)",
        {
            "sq co. private limited": "SQ & CO. PTE LTD",
            "sq co. pte. ltd.": "SQ & CO. PTE LTD",
            "sq co. pte.ltd.": "SQ & CO. PTE LTD",
            "sq co. pte ltd": "SQ & CO. PTE LTD",
        },
    ),
    (
        "Squirro",
        "squirro",
        {
            "squirro": "Squirro",
        },
    ),
]


@pytest.fixture(params=TEST_CASES)
def names2regex_cases(request):
    return request.param


def test_names2regex(names2regex_cases):
    name, expected_regex, expected_lookup = names2regex_cases
    regexes, lookup = _generate_regexes_lookup([name])
    assert expected_regex == regexes.pop()
    assert expected_lookup == lookup


class TestCompanyDataIndex:
    @classmethod
    def setup_class(cls):
        cls.index = CompanyDataIndex.load_index()

    def test_extract_companies(self):
        txt = "hello world, Company A. bla bla black sheep"
        assert ["Company A"] == self.index.extract_companies(txt)

    def test_extract_companies_similar(self):
        txt = "hello world, Squirro II. bla bla black sheep Squirro I"
        assert ["Squirro II", "Squirro I"] == self.index.extract_companies(txt)

    def test_get_all_names(self):
        expected_names = [
            "Company A",
            "Company B",
            "Company C",
            "Company D",
            "Company E",
            "Squirro I",
            "Squirro II",
        ]
        assert self.index.get_all_names() == expected_names

    def test_regexes(self):
        expected_regexes = re.compile(
            "company a|company b|company c|company d|company e|squirro i|squirro ii"
        )

        assert sorted(self.index.regex.pattern.split("|")) == sorted(
            expected_regexes.pattern.split("|")
        )

    def test_search_by_cif(self):
        result = self.index.search_by_cifs("22222")
        expected_result = [
            {
                "company_cif": ["22222"],
                "company_name": ["Company A"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_22222"],
                "country": ["SG"],
                "rm_name": ["Company A RM"],
                "permission_code": ["B1", "C1"],
                "wfi_company_rm_code": ["R011"],
                "wfi_company_team_code": ["B1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team A"],
            },
        ]
        assert result == expected_result

    def test_search_by_cifs(self):
        result = self.index.search_by_cifs(["11111", "22222"])
        expected_result = [
            {
                "company_cif": ["11111"],
                "company_name": ["Company B"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_11111"],
                "country": ["SG"],
                "permission_code": ["A1", "B1"],
                "rm_name": ["Company B RM"],
                "wfi_company_rm_code": ["R012"],
                "wfi_company_team_code": ["A1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team B"],
            },
            {
                "company_cif": ["22222"],
                "company_name": ["Company A"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_22222"],
                "country": ["SG"],
                "permission_code": ["B1", "C1"],
                "rm_name": ["Company A RM"],
                "wfi_company_rm_code": ["R011"],
                "wfi_company_team_code": ["B1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team A"],
            },
        ]
        assert result == expected_result

    def test_search_by_cif_not_found(self):
        result = self.index.search_by_cifs("000")
        expected_result = []
        assert result == expected_result

    def test_search_by_name(self):
        result = self.index.search_by_names("Company A")
        expected_result = [
            {
                "company_cif": ["22222"],
                "company_name": ["Company A"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_22222"],
                "country": ["SG"],
                "permission_code": ["B1", "C1"],
                "rm_name": ["Company A RM"],
                "wfi_company_rm_code": ["R011"],
                "wfi_company_team_code": ["B1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team A"],
            },
        ]
        assert result == expected_result

    def test_search_by_names(self):
        result = self.index.search_by_names(["Company A", "Company C"])
        expected_result = [
            {
                "company_cif": ["22222"],
                "company_name": ["Company A"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_22222"],
                "country": ["SG"],
                "permission_code": ["B1", "C1"],
                "rm_name": ["Company A RM"],
                "wfi_company_rm_code": ["R011"],
                "wfi_company_team_code": ["B1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team A"],
            },
            {
                "company_cif": ["33333"],
                "company_name": ["Company C"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_33333"],
                "country": ["SG"],
                "permission_code": ["C1", "D1"],
                "rm_name": ["Company C RM"],
                "wfi_company_rm_code": ["R013"],
                "wfi_company_team_code": ["C1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team C"],
            },
        ]
        assert result == expected_result

    def test_search_by_name_not_found(self):
        result = self.index.search_by_names("Company Z")
        expected_result = []
        assert result == expected_result

    def test_search_by_uid(self):
        result = self.index.search_by_uids("SG_55555")
        expected_result = [
            {
                "company_cif": ["55555"],
                "company_name": ["Company E"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_55555"],
                "country": ["SG"],
                "permission_code": ["E1", "F1"],
                "rm_name": ["Company E RM"],
                "wfi_company_rm_code": ["R015"],
                "wfi_company_team_code": ["E1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team E"],
            },
        ]
        assert result == expected_result

    def test_search_by_uids(self):
        result = self.index.search_by_uids(["SG_55555", "SG_44444"])
        expected_result = [
            {
                "company_cif": ["55555"],
                "company_name": ["Company E"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_55555"],
                "country": ["SG"],
                "permission_code": ["E1", "F1"],
                "rm_name": ["Company E RM"],
                "wfi_company_rm_code": ["R015"],
                "wfi_company_team_code": ["E1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team E"],
            },
            {
                "company_cif": ["44444"],
                "company_name": ["Company D"],
                "company_segment_full": ["REAL ESTATE"],
                "company_uid": ["SG_44444"],
                "country": ["SG"],
                "permission_code": ["D1", "E1"],
                "rm_name": ["Company D RM"],
                "wfi_company_rm_code": ["R014"],
                "wfi_company_team_code": ["D1"],
                "wfi_company_segment": ["R"],
                "wfi_company_team_name": ["Team D"],
            },
        ]
        assert result == expected_result

    def test_search_by_uid_not_found(self):
        result = self.index.search_by_uids("000")
        expected_result = []
        assert result == expected_result
