import pytest

from octopus.data.company_data import company_data_augmentation


def test_uid_provided(item_uid_provided, company_a_b, mock_redis_client):
    company_data_augmentation(item_uid_provided, company_a_b, skip_company_uid=True, redis_client=mock_redis_client)

    assert item_uid_provided["keywords"]["company_uid"] == ["SG_11111", "SG_22222"]
    assert item_uid_provided["keywords"]["company_cif"] == ["11111", "22222"]
    assert set(item_uid_provided["keywords"]["permission_code"]) == {
        "A1",
        "B1",
        "C1",
    }
    assert item_uid_provided["keywords"]["uid_permission_code"] == [
        "SG_11111___A1",
        "SG_11111___B1",
        "SG_22222___B1",
        "SG_22222___C1",
    ]


def test_one_company_name(item, company_a, mock_redis_client):
    item["keywords"].update({"company_name": ["Company A"]})
    company_data_augmentation(item, company_a, skip_company_uid=False, redis_client=mock_redis_client)

    assert item["keywords"]["company_uid"] == ["SG_22222"]
    assert item["keywords"]["company_cif"] == ["22222"]
    assert set(item["keywords"]["permission_code"]) == {"B1", "C1"}
    assert item["keywords"]["uid_permission_code"] == [
        "SG_22222___B1",
        "SG_22222___C1",
    ]


def test_multiple_company_names(item, company_a_b, mock_redis_client):
    item["keywords"].update({"company_name": ["Company A", "Company B"]})
    company_data_augmentation(item, company_a_b, skip_company_uid=False, redis_client=mock_redis_client)

    assert item["keywords"]["company_uid"] == ["SG_11111", "SG_22222"]
    assert item["keywords"]["company_cif"] == ["11111", "22222"]
    assert set(item["keywords"]["permission_code"]) == {"A1", "B1", "C1"}
    assert item["keywords"]["uid_permission_code"] == [
        "SG_11111___A1",
        "SG_11111___B1",
        "SG_22222___B1",
        "SG_22222___C1",
    ]


def test_multiple_company_names_same_codes(
    item, company_a_b_same_codes, mock_redis_client
):
    item["keywords"].update({"company_name": ["Company A", "Company B"]})
    company_data_augmentation(item, company_a_b_same_codes, skip_company_uid=False, redis_client=mock_redis_client)

    assert item["keywords"]["company_uid"] == ["SG_11111", "SG_22222"]
    assert item["keywords"]["company_cif"] == ["11111", "22222"]
    assert set(item["keywords"]["permission_code"]) == {"A1", "B1"}
    assert item["keywords"]["uid_permission_code"] == [
        "SG_11111___A1",
        "SG_11111___B1",
        "SG_22222___B1",
        "SG_22222___A1",
    ]


def test_multiple_company_names_different_codes(
    item, company_a_b_different_codes, mock_redis_client
):
    item["keywords"].update({"company_name": ["Company A", "Company B"]})
    company_data_augmentation(
        item, company_a_b_different_codes, skip_company_uid=False, redis_client=mock_redis_client
    )

    assert item["keywords"]["company_uid"] == ["SG_11111", "SG_22222"]
    assert item["keywords"]["company_cif"] == ["11111", "22222"]
    assert set(item["keywords"]["permission_code"]) == {"A1", "B1", "C1", "D1"}
    assert item["keywords"]["uid_permission_code"] == [
        "SG_11111___A1",
        "SG_11111___B1",
        "SG_22222___C1",
        "SG_22222___D1",
    ]


def test_company_data_missing_key(item, company_a_missing_key, mock_redis_client):
    item["keywords"].update({"company_name": ["Company A"]})
    company_data_augmentation(item, company_a_missing_key, skip_company_uid=False, redis_client=mock_redis_client)

    assert item["keywords"] == {
        "company_cif": [],
        "company_name": ["Company A"],
        "company_uid": [],
        "rm_name": ["Company A RM"],
    }
    mock_redis_client.sadd.assert_called_with("company_data_missing_keys", "Company A")


def test_no_permission_codes(item, company_a, mock_redis_client):
    company_a[0].pop("permission_code")
    item["keywords"].update({"company_name": ["Company A"]})
    company_data_augmentation(item, company_a, skip_company_uid=False, redis_client=mock_redis_client)

    assert item["keywords"] == {
        "company_cif": ["22222"],
        "company_uid": ["SG_22222"],
        "company_name": ["Company A"],
        "rm_name": ["Company A RM"],
    }
    mock_redis_client.sadd.assert_called_with(
        "company_data_missing_permission_codes", "Company A"
    )


@pytest.fixture()
def item():
    return {
        "id": "test-item-1",
        "title": "Test Item",
        "keywords": {},
    }


@pytest.fixture()
def item_uid_provided():
    return {
        "id": "test-item-1",
        "title": "Test Item",
        "keywords": {
            "company_name": [
                "Company B",
                "Company A",
            ],
            "company_uid": ["SG_11111", "SG_22222"],
        },
    }


@pytest.fixture()
def company_a():
    return [
        {
            "company_cif": ["22222"],
            "company_name": ["Company A"],
            "company_uid": ["SG_22222"],
            "permission_code": ["B1", "C1"],
            "rm_name": ["Company A RM"],
        }
    ]


@pytest.fixture()
def company_a_missing_key():
    return [
        {
            "company_name": ["Company A"],
            "company_uid": ["SG_22222"],
            "permission_code": ["B1", "C1"],
            "rm_name": ["Company A RM"],
        }
    ]


@pytest.fixture()
def company_a_b():
    return [
        {
            "company_cif": ["11111"],
            "company_name": ["Company B"],
            "company_uid": ["SG_11111"],
            "permission_code": ["A1", "B1"],
            "rm_name": ["Company B RM"],
        },
        {
            "company_cif": ["22222"],
            "company_name": ["Company A"],
            "company_uid": ["SG_22222"],
            "permission_code": ["B1", "C1"],
            "rm_name": ["Company A RM"],
        },
    ]


@pytest.fixture()
def company_a_b_same_codes():
    return [
        {
            "company_cif": ["11111"],
            "company_name": ["Company B"],
            "company_uid": ["SG_11111"],
            "permission_code": ["A1", "B1"],
            "rm_name": ["Company B RM"],
        },
        {
            "company_cif": ["22222"],
            "company_name": ["Company A"],
            "company_uid": ["SG_22222"],
            "permission_code": ["B1", "A1"],
            "rm_name": ["Company A RM"],
        },
    ]


@pytest.fixture()
def company_a_b_different_codes():
    return [
        {
            "company_cif": ["11111"],
            "company_name": ["Company B"],
            "company_uid": ["SG_11111"],
            "permission_code": ["A1", "B1"],
            "rm_name": ["Company A RM"],
        },
        {
            "company_cif": ["22222"],
            "company_name": ["Company A"],
            "company_uid": ["SG_22222"],
            "permission_code": ["C1", "D1"],
            "rm_name": ["Company A RM"],
        },
    ]
