from unittest.mock import MagicMock

import pytest
from requests import Response
from requests.exceptions import HTTPError

from octopus.clients import init_wfi_client
from octopus.clients.wfi_client import WFIClient


def test_init_wfi_client():
    wfi_client = init_wfi_client()

    assert wfi_client.base_url == "https://test-url@example.com:1234/wfiapi"
    assert wfi_client.session is not None
    wfi_client.session.headers.update.assert_called_with(
        {
            "domain": "test-domain",
            "objectstore": "test-objectstore",
            "channel": "test-channel",
            "test-api-header-name": "test-api-key",
        }
    )


class TestWFIClient:
    def setup_method(self):
        self.wfi_client = WFIClient(
            base_url="https://test-url@example.com:1234/wfiapi",
            headers={
                "domain": "test-domain",
                "objectstore": "test-objectstore",
                "channel": "test-channel",
                "test-api-header-name": "test-api-key",
            },
        )
        self.files = {"File": b"test"}
        self.data = {"key": "value"}

    def test_checkin_document(self):
        mock_response = MagicMock(spec=Response)
        mock_response.status_code = 200
        mock_response.text = "{12345-67890}"
        self.wfi_client.session.post.return_value = mock_response
        doc_id = self.wfi_client.checkin_document(self.files, self.data)

        assert doc_id == "12345-67890"
        self.wfi_client.session.post.assert_called_once_with(
            f"{self.wfi_client.base_url}/checkindocument",
            files=self.files,
            data=self.data,
        )

    def test_checkin_document_minor_version(self):
        orig_doc_id = "12345-67890"
        mock_response = MagicMock(spec=Response)
        mock_response.status_code = 200
        mock_response.text = "{12345-67891}"
        self.wfi_client.session.post.return_value = mock_response
        new_doc_id = self.wfi_client.checkin_document(
            self.files,
            self.data,
            document_id=orig_doc_id,
        )

        assert new_doc_id == "12345-67891"
        self.wfi_client.session.post.assert_called_once_with(
            f"{self.wfi_client.base_url}/checkindocument/{{{orig_doc_id}}}/d",
            files=self.files,
            data=self.data,
        )

    def test_checkin_document_failed(self):
        mock_response = MagicMock(spec=Response)
        mock_response.raise_for_status.side_effect = HTTPError
        self.wfi_client.session.post.return_value = mock_response

        with pytest.raises(HTTPError):
            self.wfi_client.checkin_document(
                self.files,
                self.data,
            )

    def test_fetch_metadata(self):
        doc_id = "12345-67890"
        sample_data = ["data1", "data2", "data3"]
        mock_response = MagicMock(spec=Response)
        mock_response.json.return_value = sample_data
        self.wfi_client.session.get.return_value = mock_response
        data = self.wfi_client.fetch_metadata("12345-67890")

        assert data == sample_data
        self.wfi_client.session.get.assert_called_once_with(
            f"{self.wfi_client.base_url}/fetchmetadatabydocid/{{{doc_id}}}"
        )

    def test_fetch_metadata_with_additional_fields(self):
        doc_id = "12345-67890"
        sample_data = ["data1", "data2", "data3", "others"]
        mock_response = MagicMock(spec=Response)
        mock_response.json.return_value = sample_data
        self.wfi_client.session.post.return_value = mock_response
        data = self.wfi_client.fetch_metadata("12345-67890", fields=["others"])

        assert data == sample_data
        self.wfi_client.session.post.assert_called_once_with(
            f"{self.wfi_client.base_url}/fetchmetadatabydocid/{{{doc_id}}}",
            json=["others"],
        )
