"""Squirro storage related utilities."""

import logging
from pathlib import Path
from typing import TYPE_CHECKING

from squirro.common.config import get_config

try:
    from squirro.lib.storage.handler import StorageHandler
except Exception:
    logging.exception("Failed to import StorageHandler.")

if TYPE_CHECKING:
    from io import BytesIO
    from typing import Any


def get_pdf_url(item: "dict[str, Any]") -> "str | None":
    """Get the pdf url of the item.

    Args:
        item: The item.

    Returns:
        The pdf url.
    """
    for file in item.get("files", []):
        if file.get("mime_type", "") == "application/pdf":
            url: str = file.get("content_url", "")
            return url
    return None


def get_pdf_full_path(item: "dict[str, Any]") -> "Path | None":
    """Get the full path of the item.

    Args:
        item: The item.

    Returns:
        The full path to the pdf file.
    """
    if not (url := get_pdf_url(item)):
        return None
    return get_storage_url_full_path(url)


def get_storage_url_full_path(storage_url: str) -> Path:
    """Get the full path of a file.

    Args:
        storage_url: The storage url.

    Returns:
        The full path to the file.
    """
    config = get_config("squirro.lib.storage")
    try:
        # pylint: disable-next=protected-access
        bucket, _, path = StorageHandler(config)._parse_url(storage_url)  # noqa: SLF001
        return Path(config.get(f"storage_{bucket}", "directory"), path)
    except Exception:
        logging.exception("Failed to parse content URL.")
        raise


def open_pdf(item: "dict[str, Any]") -> "BytesIO | None":
    """Open the pdf of the item using StorageHandler.

    Args:
        item: The item.

    Returns:
        The pdf file.
    """
    if not (url := get_pdf_url(item)):
        return None

    config = get_config("squirro.lib.storage")
    try:
        pdf: BytesIO = StorageHandler(config).open(url)
    except Exception:
        logging.exception("Failed to open pdf file.")
        raise
    else:
        return pdf
