"""Encryption related utilities."""

import json
import logging
from typing import TYPE_CHECKING

from cryptography.fernet import Fernet

from .files import ensure_file_exists

if TYPE_CHECKING:
    from typing import Any


def decrypt(value: str, key: str) -> str:
    """Decrypt value with key.

    Args:
        value: Value to decrypt.
        key: Key to decrypt value with.

    Returns:
        Decrypted value.
    """
    return Fernet(key).decrypt(value.encode("utf-8")).decode("utf-8")


def encrypt(value: str, key: str) -> str:
    """Encrypt value with key.

    Args:
        value: Value to encrypt.
        key: Key to encrypt value with.

    Returns:
        Encrypted value.
    """
    return Fernet(key).encrypt(value.encode("utf-8")).decode("utf-8")


def load_key(filepath: str = "/etc/squirro/.squirro.key") -> str:
    """Load key from file.

    Args:
        filepath: Path to key file.

    Returns:
        Key.
    """
    path = ensure_file_exists(filepath)
    return path.read_text("utf-8")


def load_vault(
    filepath: str = "/etc/squirro/.vault.json",
) -> "dict[str, dict[str, Any]]":
    """Load vault from file.

    Args:
        filepath: Path to vault file.

    Returns:
        Vault.

    Raises:
        JSONDecodeError: If the vault file is not valid JSON.
    """
    path = ensure_file_exists(filepath)

    try:
        data: dict[str, dict[str, str]] = json.loads(path.read_text("utf-8"))
    except json.JSONDecodeError:
        logging.exception("Vault file is not valid JSON: %s", path)
        raise
    else:
        return data
