"""Upload structures."""

import json
from dataclasses import dataclass
from pathlib import Path
from typing import TYPE_CHECKING

from octopus.utils import remove_dict_nested_empty_values

if TYPE_CHECKING:
    from typing import Any


@dataclass
class UploadPayload(dict):  # type: ignore[type-arg]
    """Upload payload."""

    file_paths: "list[Path]"
    invalids: "list[dict[str, Any]]"
    labels: "dict[str, Any]"
    name: str
    email: str

    def __init__(self) -> None:
        """Initialize the upload payload."""
        self.file_paths = []
        self.invalids = []
        self.labels = {}
        self.name = ""
        self.email = ""

    @classmethod
    def create_payload(cls, **kwargs: "dict[str, Any]") -> "UploadPayload":
        """Initialize the upload payload from a dictionary.

        Args:
            kwargs: Dictionary to initialize the upload payload from.

        Returns:
            Upload payload.
        """
        obj = cls()
        remove_dict_nested_empty_values(kwargs)
        for k, v in kwargs.items():
            setattr(obj, k, v)
        return obj

    @classmethod
    def from_dict(cls, data: "dict[str, Any]") -> "UploadPayload":
        """Initialize the upload payload from a dictionary.

        Args:
            data: Dictionary to initialize the upload payload from.

        Returns:
            Upload payload.
        """
        obj = cls()
        for k, v in data.items():
            if k in {"file_paths", "invalids", "labels"}:
                if k == "file_paths":
                    setattr(obj, k, [Path(path) for path in json.loads(v)])
                else:
                    setattr(obj, k, json.loads(v))
            else:
                setattr(obj, k, v)
        return obj

    def to_dict(self) -> "dict[str, Any]":
        """Convert to dictionary.

        Returns:
            Dictionary representation of the upload payload.
        """
        data = self.__dict__
        for k in ("file_paths", "invalids", "labels"):
            if v := data.get(k):
                if k == "file_paths":
                    v = [str(path) for path in v]
                data[k] = json.dumps(v)
        return data
