"""Autocomplete utilities."""

import json
import logging
from pathlib import Path
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from typing import Any


class TrieNode:
    """A node in the trie structure."""

    # Each node contains a dictionary of children where each key is a character
    _children: "dict[str, TrieNode]"

    # If the node represents the end of a word
    _is_end_of_word: bool

    # Set of values that the node represents
    _words: "set[str]"

    def __init__(self) -> None:
        """Initialize the trie node."""
        self._children = {}
        self._is_end_of_word = False
        self._words = set()

    def add_word(self, value: str) -> None:
        """Add a value to the set of values that the node represents.

        Args:
            value: The value to add to the set.
        """
        self._words.add(value)

    @property
    def children(self) -> "dict[str, TrieNode]":
        """Return the children of the node."""
        return self._children

    @children.setter
    def children(self, value: "dict[str, TrieNode]") -> None:
        """Set the children of the node.

        Args:
            value: The new children of the node.
        """
        self._children = value

    @classmethod
    def from_dict(cls, data: "dict[str, Any]") -> "TrieNode":
        """Create a trie node from a dictionary.

        Args:
            data: The dictionary to create the trie node from.

        Returns:
            The created trie node.
        """
        obj = cls()
        obj.children = {k: TrieNode.from_dict(v) for k, v in data["children"].items()}
        obj.is_end_of_word = data["is_end_of_word"]
        obj.words = set(data["words"])
        return obj

    @property
    def is_end_of_word(self) -> bool:
        """Does the node represents the end of a word."""
        return self._is_end_of_word

    @is_end_of_word.setter
    def is_end_of_word(self, value: bool) -> None:
        """Set the flag to indicate if the node represents the end of a word.

        Args:
            value: The new value of the flag.
        """
        self._is_end_of_word = value

    def to_dict(self) -> "dict[str, Any]":
        """Convert the trie node to a dictionary.

        Returns:
            The dictionary representation of the trie node.
        """
        return {
            "children": {k: v.to_dict() for k, v in self._children.items()},
            "is_end_of_word": self._is_end_of_word,
            "words": list(self._words),
        }

    @property
    def words(self) -> "set[str]":
        """Return the set of values that the node represents."""
        return self._words

    @words.setter
    def words(self, value: "set[str]") -> None:
        """Set the set of values that the node represents.

        Args:
            value: The new set of values that the node represents.
        """
        self._words = value


class Autocomplete:
    """A trie data structure to store strings and perform autocomplete queries.

    The trie is case-insensitive and can store multiple words with the
    same prefix. It also has a cache to store the results of previous
    queries to improve performance.
    """

    # Cache to store results of previous queries
    _cache: "dict[str, set[str]]"

    # Keep track of the order of access for LRU
    _cache_access_order: "list[str]"

    # List of items to perform search on
    _items: "set[str]"

    # Maximum size of the cache
    _max_cache_size: int

    # Name of the autocomplete object
    _name: str

    # Root node of the trie
    _root: TrieNode

    def __init__(self, name: str, max_cache_size: int = 100) -> None:
        """Initializes the trie with an empty root node and an empty cache.

        Args:
            name: The name of the autocomplete object.
            max_cache_size: The maximum size of the cache
        """
        self._cache = {}
        self._cache_access_order = []
        self._max_cache_size = max_cache_size
        self._name = name
        self._items = set()
        self._root = TrieNode()

    def __call__(self, prefix: str) -> "set[str]":
        """Perform autocomplete search on the given prefix.

        Args:
            prefix: The prefix to search for.

        Returns:
            The set of words that start with the given prefix.
        """
        prefix = prefix.lower()
        if prefix in self._cache:
            # Move the accessed key to the end to indicate recent use
            logging.info("Cache hit for prefix %s", prefix)
            self._cache_access_order.remove(prefix)
            self._cache_access_order.append(prefix)
            return self._cache[prefix]

        results: set[str] = set()
        if not prefix:
            logging.info("Empty prefix, returning all items")
            results = self._items
        elif self._search_prefix(self._root, prefix, results):
            logging.info("Prefix match for %s, %d results", prefix, len(results))
            self._update_cache(prefix, results)
        else:
            logging.info("No prefix match for %s, performing substring match", prefix)
            results = {word for word in self._items if prefix in word.lower()}
            self._update_cache(prefix, results)
        return results

    @property
    def cache(self) -> "dict[str, set[str]]":
        """Return the cache."""
        return self._cache

    @cache.setter
    def cache(self, value: "dict[str, set[str]]") -> None:
        """Set the cache."""
        self._cache = value

    @property
    def cache_access_order(self) -> "list[str]":
        """Return the cache access order."""
        return self._cache_access_order

    @cache_access_order.setter
    def cache_access_order(self, value: "list[str]") -> None:
        """Set the cache access order."""
        self._cache_access_order = value

    def insert(self, name: str) -> None:
        """Insert a new word into the trie.

        Args:
            name: The word to insert into the autocomplete trie.
        """
        logging.info("Adding %s into the list of autocomplete items", name)
        self._items.add(name)

        # Names can have multiple word and prefix should match the name
        # irrespective of the word order
        # Eg: "Michael Johnson" and "Jessica Miller" should match for prefix "mi"
        for word in name.split():
            node = self._root
            for char in word.lower():
                if char not in node.children:
                    node.children[char] = TrieNode()
                node = node.children[char]
            node.is_end_of_word = True
            node.add_word(name)

        # Invalidate cache as new word is added
        self._cache = {}
        self._cache_access_order = []

    @property
    def items(self) -> "set[str]":
        """Return the items."""
        return self._items

    @items.setter
    def items(self, value: "set[str]") -> None:
        """Set the items."""
        self._items = value

    @classmethod
    def load_index(cls, path: str) -> "Autocomplete":
        """Loads the index from a file.

        Args:
            path: Path to load the index from.

        Returns:
            Autocomplete object.
        """
        with Path(path).open(encoding="utf-8") as f:
            data = json.load(f)
        obj = cls(data["name"])
        obj.cache = data["cache"]
        obj.cache_access_order = data["cache_access_order"]
        obj.items = set(data["items"])
        obj.max_cache_size = data["max_cache_size"]
        obj.root.from_dict(data["root"])
        return obj

    @property
    def max_cache_size(self) -> int:
        """Return the maximum cache size."""
        return self._max_cache_size

    @max_cache_size.setter
    def max_cache_size(self, value: int) -> None:
        """Set the maximum cache size."""
        self._max_cache_size = value

    @property
    def root(self) -> TrieNode:
        """Return the root node."""
        return self._root

    @root.setter
    def root(self, value: TrieNode) -> None:
        """Set the root node."""
        self._root = value

    def save_index(self, path: str) -> None:
        """Saves the index to a file.

        Args:
            path: Path to save the index.
        """
        with Path(path).open("w", encoding="utf-8") as f:
            json.dump(
                {
                    "cache": self._cache,
                    "cache_access_order": self._cache_access_order,
                    "items": list(self._items),
                    "max_cache_size": self._max_cache_size,
                    "name": self._name,
                    "root": self._root.to_dict(),
                },
                f,
            )

    def _dfs(self, node: TrieNode, prefix: str, results: "set[str]") -> None:
        """Perform a depth-first search.

        Find all words that start with the
        given prefix.

        Args:
            node: The current node in the trie.
            prefix: The prefix to search for.
            results: The set to store the results.
        """
        if node.is_end_of_word:
            results.update(node.words)
        for char in node.children:
            self._dfs(node.children[char], prefix + char, results)

    def _search_prefix(self, node: TrieNode, prefix: str, results: "set[str]") -> bool:
        """Search using prefix.

        Search for the node that represents the last character in the
        prefix.

        Args:
            node: The current node in the trie.
            prefix: The prefix to search for.
            results: The set to store the results.

        Returns:
            True if the prefix is found, False otherwise.
        """
        for char in prefix:
            if char in node.children:
                node = node.children[char]
            else:
                return False
        self._dfs(node, prefix, results)
        return True

    def _update_cache(self, key: str, value: "set[str]") -> None:
        """Update the cache with the new key-value pair.

        Least recently used item is removed if the cache size exceeds the
        maximum size.

        Args:
            key: The key to add to the cache.
            value: The value to add to the cache.
        """
        self._cache[key] = value
        self._cache_access_order.append(key)

        if len(self._cache_access_order) > self._max_cache_size:
            lru_key = self._cache_access_order.pop(0)
            logging.info("Evicting least recently used key %s from the cache", lru_key)
            del self._cache[lru_key]
