"""Stream utilities."""

import logging
import os
from typing import TYPE_CHECKING

from redis import ResponseError

from octopus.clients import inject_redis_client
from octopus.utils import remove_dict_nested_empty_values

from .exceptions import AddToStreamError

if TYPE_CHECKING:
    from typing import Any

    from redis import StrictRedis

    from ._types import StreamMessagesType


RETRY_COUNT = 3


@inject_redis_client
def add_to_stream(
    payload: "dict[str, Any]",
    stream: str,
    **kwargs: "Any",  # noqa: ANN401
) -> None:
    """Add payload to stream.

    Args:
        payload: Payload to add to stream.
        stream: Stream to add payload to.
        **kwargs: Keyword arguments.

    Raises:
        AddToStreamError: If adding to stream fails.
    """
    redis_client: StrictRedis[bytes] = kwargs["redis_client"]
    remove_dict_nested_empty_values(payload)

    retry = 0
    while retry < RETRY_COUNT:
        try:
            redis_client.xadd(stream, payload, maxlen=1000)
            break
        except Exception:
            logging.exception(
                "Attempt #%d failed. Error adding payload to `%s` stream.",
                retry + 1,
                stream,
            )
            retry += 1
            continue
    else:
        raise AddToStreamError(stream)


@inject_redis_client
def create_consumer_group(
    stream_name: str,
    group_name: str,
    **kwargs: "Any",  # noqa: ANN401
) -> None:
    """Create consumer group for email stream if it does not exist.

    Args:
        stream_name: Stream to create consumer group for.
        group_name: Name of the consumer group.
        **kwargs: Keyword arguments.
    """
    redis_client: StrictRedis[bytes] = kwargs["redis_client"]

    try:
        redis_client.xgroup_create(stream_name, group_name, mkstream=True)
        logging.info("Consumer group created.")
    except ResponseError:
        logging.info("Consumer group already exists.")


@inject_redis_client
def read_stream(
    stream_name: str,
    group_name: str,
    **kwargs: "Any",  # noqa: ANN401
) -> "list[StreamMessagesType]":
    """Read payloads from stream.

    Args:
        stream_name: Stream to read from.
        group_name: Name of the consumer group.
        **kwargs: Keyword arguments.

    Returns:
        List of messages from the stream.
    """
    redis_client: StrictRedis[bytes] = kwargs["redis_client"]

    # '>' tells Redis to return only the new messages added to the stream
    messages: list[StreamMessagesType] = redis_client.xreadgroup(
        group_name,
        os.uname()[1],
        {stream_name: ">"},
        count=10,
        block=3000,
    )
    return messages
