"""Streamer."""

import logging
import time
from abc import abstractmethod
from typing import TYPE_CHECKING

from octopus.clients import init_redis_client

from .utils import create_consumer_group, read_stream

if TYPE_CHECKING:
    from typing import Any

    from octopus.stream import StreamMessagesType


class Streamer:
    """Base streamer class."""

    def __init__(self, stream_name: str, group_name: str) -> None:
        """Initialize streamer.

        Args:
            stream_name: Name of the stream.
            group_name: Name of the consumer group.
        """
        self.stream_name = stream_name
        self.group_name = group_name

        self.redis_client = init_redis_client()
        create_consumer_group(stream_name, group_name, redis_client=self.redis_client)

    def run(self) -> None:
        """Run streamer."""
        while True:
            try:
                messages = read_stream(
                    self.stream_name, self.group_name, redis_client=self.redis_client
                )
            except Exception:
                logging.exception(
                    "Connection error reading. Reconnecting after 3 seconds..."
                )
                time.sleep(3)
                continue

            if not messages:
                continue

            self.preprocess()
            for message in messages:
                try:
                    self._process(message)
                except Exception:
                    logging.exception(
                        "Connection error acknowledging. "
                        "Reconnecting after 3 seconds..."
                    )
                    time.sleep(3)
            self.postprocess()

    def _process(self, message: "StreamMessagesType") -> None:
        """Process message."""
        logging.debug("Processing message: %s", message)
        for bmid, bpayload in message[1]:
            mid: str = bmid.decode("utf-8")  # type: ignore[union-attr]
            logging.debug("Processing message %s", mid)
            if not isinstance(bpayload, dict):
                logging.exception("Message ID %s has invalid message format.", mid)
                continue
            payload = {
                k.decode("utf-8"): v.decode("utf-8") for k, v in bpayload.items()
            }
            try:
                self.process(payload)
            except Exception:
                logging.exception("Failed to process message %s", mid)
                continue
            self.redis_client.xack(self.stream_name, self.group_name, mid)

    @abstractmethod
    def process(self, payload: "dict[str, Any]") -> None:
        """Process message.

        Args:
            payload: Payload to process.
        """

    @abstractmethod
    def preprocess(self) -> None:
        """Preprocess."""

    @abstractmethod
    def postprocess(self) -> None:
        """Postprocess."""
