"""PDF utilities."""

import logging
import subprocess  # noqa: S404 - Used to call `ghostscript`
from io import BytesIO


def fix_invalid_pdf(pdf_stream: BytesIO) -> BytesIO:
    """Fixes a PDF file that is invalid due to missing endstream/endobj tags.

    Args:
        pdf_stream: A file-like object containing the PDF data.

    Returns:
        A file-like object containing the fixed PDF data.

    Raises:
        CalledProcessError: If the subprocess fails.
        RuntimeError: If the Ghostscript command fails.
    """
    try:
        # Run the Ghostscript command with input and output set to pipes
        gs_command = [
            "gs",
            "-q",
            "-o",
            "-",
            "-sDEVICE=pdfwrite",
            "-",
        ]

        res = subprocess.run(  # noqa: S603
            gs_command,
            capture_output=True,
            shell=False,
            input=pdf_stream.read(),
            check=True,
        )
        if res.returncode != 0:
            msg = f"Ghostscript error: {res.stderr.decode()}"
            logging.error(msg)
            raise RuntimeError(msg)
        return BytesIO(res.stdout)
    except subprocess.CalledProcessError:
        logging.exception("Ghostscript error")
        raise
