"""Check if a PDF is password protected."""

import logging
from io import BytesIO

from pdfminer.pdfdocument import PDFDocument, PDFPasswordIncorrect
from pdfminer.pdfparser import PDFParser, PDFSyntaxError

from .utils import fix_invalid_pdf


def check_password_protected_pdf(fs: BytesIO) -> "bool":
    """Checks if a PDF file is password protected.

    Args:
        fs: A file-like object containing the PDF data.

    Returns:
        True if the PDF requires a password to open, False otherwise
    """

    def func(f: BytesIO) -> "bool":
        """Checks if a PDF requires a password to be opened."""
        try:
            parser = PDFParser(f)
            PDFDocument(parser, password="")
        except PDFSyntaxError:
            logging.warning("PDF is invalid, attempting to fix it")
            return func(fix_invalid_pdf(f))  # noqa: DOC201
        except PDFPasswordIncorrect:
            logging.warning("Unable to open, PDF is password protected")
            return True
        else:
            return False

    return func(fs)
