"""Email structures."""

import json
from dataclasses import dataclass
from pathlib import Path
from typing import TYPE_CHECKING

from octopus.utils import load_config, remove_dict_nested_empty_values

if TYPE_CHECKING:
    from typing import Any


@dataclass
class EmailConfig(dict):  # type: ignore[type-arg]
    """Email config."""

    distribution_list: dict[str, str]
    triggers: dict[str, list[str]]

    def to_dict(self) -> "dict[str, Any]":
        """Convert to dictionary.

        Returns:
            Dictionary representation of the email config.
        """
        return self.__dict__


@dataclass
class EmailPayload(dict):  # type: ignore[type-arg]
    """Email payload."""

    data: "dict[str, Any]"
    recipients: str
    subject: str
    template: str
    type: str

    def __init__(self) -> None:
        """Initialize the email payload."""
        self.data = {}

    @classmethod
    def create_payload(cls, **kwargs: "dict[str, Any]") -> "EmailPayload":
        """Initialize the email payload from a dictionary.

        Args:
            kwargs: Dictionary to initialize the email payload from.

        Returns:
            Email payload.
        """
        obj = cls()
        data: dict[str, Any] = kwargs.pop("data")
        remove_dict_nested_empty_values(data)
        for k, v in kwargs.items():
            setattr(obj, k, v)
        cfg = load_config()
        data["show_disclaimer"] = cfg["squirro"]["environment"] != "PROD"
        obj.data = data

        template_dir = Path("/opt/squirro/octopus/assets/email_templates")
        if obj.type == "ingestion_report":
            obj.subject = "OCtopus Ingestion Update"
            obj.template = str(template_dir / "ingestion_report.html")
        elif obj.type == "status_update":
            obj.subject = "OCtopus Status Update"
            obj.template = str(template_dir / "status_tracking.html")

        return obj

    @classmethod
    def from_dict(cls, data: "dict[str, Any]") -> "EmailPayload":
        """Initialize the email payload from a dictionary.

        Args:
            data: Dictionary to initialize the email payload from.

        Returns:
            Email payload.
        """
        obj = cls()
        for k, v in data.items():
            if k == "data":
                setattr(obj, k, json.loads(v))
            else:
                setattr(obj, k, v)
        return obj

    def to_dict(self) -> "dict[str, Any]":
        """Convert to dictionary.

        Returns:
            Dictionary representation of the email payload.
        """
        data = self.__dict__
        data["data"] = json.dumps(data["data"])
        return data
