"""Company data tagging."""

import json
import logging
from typing import TYPE_CHECKING

from octopus.clients.wfi_client import WFI_FIELDS
from octopus.utils import prettier_dict

if TYPE_CHECKING:
    from collections.abc import Hashable
    from typing import Any

    from redis import Redis


COMPANY_DATA_LABELS: "set[str]" = {
    "wfi_company_cif",
    "wfi_company_name",
    "wfi_company_rm_code",
    "wfi_company_segment",
    "wfi_company_team_name",
}

DOCUMENT_TYPE_LABELS = {
    "wfi_document_category",
    "wfi_document_name",
    "wfi_document_type",
}


def tag_company_data(
    item: "dict[str, Any]",
    company: "dict[Hashable, list[str]]",
) -> None:
    """Tag company data.

    Args:
        item: The item to tag.
        company: The company data.
    """
    uid = company.get("company_uid", [""])[0]
    data = {label: WFI_FIELDS[label]["default"] for label in COMPANY_DATA_LABELS}

    if not uid:
        logging.warning("No company uid found")
        item["keywords"].update(data)
        return

    for label in COMPANY_DATA_LABELS:
        if label in {"wfi_company_cif", "wfi_company_name"}:
            sq_label = label.lstrip("wfi_")
            val = item.get("keywords", {}).get(sq_label, WFI_FIELDS[label]["default"])
        else:
            val = company.get(label, WFI_FIELDS[label]["default"])
        data[label] = val[:1]

    item["keywords"].update(data)
    logging.info("Tagged company data %s", prettier_dict(data))


def tag_document_type_data(
    item: "dict[str, Any]",
    redis_client: "Redis[bytes]",
) -> None:
    """Tag document type data.

    Args:
        item: The item to tag.
        redis_client: The Redis client.
    """
    document_type = item["keywords"].get("document_type", [""])[0].upper()
    data = {label: WFI_FIELDS[label]["default"] for label in DOCUMENT_TYPE_LABELS}
    mapping_raw: bytes | None = redis_client.hget(
        "document_type_mapping_hash", document_type
    )
    if mapping_raw:
        mapping = json.loads(mapping_raw)
        data.update(
            {
                key: [str(value).lower() if key == "bbca_documents" else value]
                for key, value in mapping.items()
            }
        )

    item["keywords"].update(data)
    logging.info("Tagged document data %s", prettier_dict(data))
