"""Company data augmentation."""

import logging
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from collections.abc import Hashable
    from typing import Any

    from redis import Redis


def company_data_augmentation(
    item: "dict[str, Any]",
    company_data: "list[dict[Hashable, list[str]]]",
    *,
    skip_company_uid: bool,
    redis_client: "Redis[bytes]",
) -> None:
    """Tag documents with uid, cif, and permission codes.

    Retrieves the entitlements of a company and assigns these as
    labels to the documents.

    Args:
        item: The item to tag
        company_data: The company data
        skip_company_uid: Whether to skip the company uid
        redis_client: The redis client
    """
    cifs: list[str] = []
    permission_codes = set()
    uids = item.get("keywords", {}).get("company_uid", [])
    uid_permission_codes: list[str] = []
    rm_name: list[str] = []

    for company in company_data:
        name = company["company_name"][0]
        if rm := company.get("rm_name"):
            rm_name.extend(rm)

        try:
            cif = company["company_cif"][0]
            codes = company.get("permission_code", [])
            uid = company["company_uid"][0]
            if not codes and name:
                logging.exception("`%s` missing permission codes", name)
                redis_client.sadd("company_data_missing_permission_codes", name)
        except KeyError:
            logging.exception("`%s` missing keys", name)
            redis_client.sadd("company_data_missing_keys", name)
            continue

        if not skip_company_uid:
            uids.append(uid)
        cifs.append(cif)

        # Add the uid as the prefix to each permission code
        uid_permission_codes.extend([f"{uid}___{x}" for x in codes])
        permission_codes |= set(codes)

    if rm_name:
        item["keywords"]["rm_name"] = rm_name[:1]  # Only set the rm_name once
    item["keywords"]["company_uid"] = uids
    item["keywords"]["company_cif"] = cifs
    if permission_codes:
        item["keywords"]["permission_code"] = list(permission_codes)
        item["keywords"]["uid_permission_code"] = uid_permission_codes

    logging.info(
        "Company data augmentation: %s, %s, %s, %s",
        item["keywords"].get("company_uid", []),
        item["keywords"].get("company_cif", []),
        item["keywords"].get("permission_code", []),
        item["keywords"].get("rm_name", []),
    )
