"""WFI client that interacts with OCBC's filesystem."""

import logging
import re
from datetime import datetime
from typing import TYPE_CHECKING

import pytz
import requests
from requests.adapters import HTTPAdapter
from requests.exceptions import ContentDecodingError, InvalidHeader

from octopus.utils import load_config

if TYPE_CHECKING:
    from collections.abc import Hashable
    from configparser import ConfigParser
    from typing import Any

EMPTY = [""]
NA = ["NA"]
UNASSIGNED = ["unassigned"]
UNCLASSIFIED = ["Unclassified"]
TODAY = [datetime.now(tz=pytz.timezone("Asia/Singapore")).strftime("%Y-%m-%d")]

# Case-Insensitive, lower-case for standardization
STRING = "string"
BOOLEAN = "boolean"

WFI_FIELDS = {
    "wfi_company_cif": {
        "default": EMPTY,
        "length": 15,
        "wfi": {
            "propertyName": "BBCA_CIF",
            "propertyDataType": STRING,
        },
    },
    "wfi_company_name": {
        "default": EMPTY,
        "length": 150,
        "wfi": {
            "propertyName": "BBCA_CustomerName",
            "propertyDataType": STRING,
        },
    },
    "wfi_company_segment": {
        "default": UNASSIGNED,
        "length": 64,
        "wfi": {
            "propertyName": "BBCA_Segment",
            "propertyDataType": STRING,
        },
    },
    "wfi_company_rm_code": {
        "default": NA,
        "length": 64,
        "wfi": {
            "propertyName": "BBCA_BusinessUnit",
            "propertyDataType": STRING,
        },
    },
    "wfi_company_team_name": {
        "default": NA,
        "length": 64,
        "wfi": {
            "propertyName": "BBCA_Team",
            "propertyDataType": STRING,
        },
    },
    "wfi_document_category": {
        "default": UNCLASSIFIED,
        "length": 250,
        "wfi": {
            "propertyName": "BBCA_DocumentCategory",
            "propertyDataType": STRING,
        },
    },
    "wfi_document_name": {
        "default": UNCLASSIFIED,
        "length": 250,
        "wfi": {
            "propertyName": "BBCA_DocumentName",
            "propertyDataType": STRING,
        },
    },
    "wfi_document_type": {
        "default": UNCLASSIFIED,
        "length": 250,
        "wfi": {
            "propertyName": "BBCA_DocumentType",
            "propertyDataType": STRING,
        },
    },
    "wfi_document_date": {
        "default": TODAY,
        "length": 64,
        "wfi": {
            "propertyName": "BBCA_DocumentDate",
            "propertyDataType": "DATE",
            "format": "YYYY-MM-DD",
        },
    },
    "wfi_references": {
        "default": EMPTY,
        "length": 250,
        "wfi": {
            "propertyName": "BBCA_Reference",
            "propertyDataType": STRING,
        },
    },
}


def init_wfi_headers(
    cfg: "ConfigParser | None" = None,
) -> "tuple[str, dict[str, str]]":
    """Initialize WFI headers.

    Args:
        cfg: ConfigParser instance. Defaults to None.

    Returns:
        Tuple containing the base url and headers.
    """
    if not cfg:
        cfg = load_config()

    port = f":{cfg['wfi']['port']}" if cfg["wfi"]["port"] else ""
    base_url = f"{cfg['wfi']['url']}{port}/wfiapi"

    wfi_headers = {
        "domain": cfg["wfi"]["domain"],
        "objectstore": cfg["wfi"]["objectstore"],
        "channel": cfg["wfi"]["channel"],
        cfg["wfi"]["wfi_api_header_name"]: cfg["wfi"]["api_key"],
    }

    return base_url, wfi_headers


def init_wfi_client(
    cfg: "ConfigParser | None" = None,
) -> "WFIClient":
    """Initialize wfi client.

    Args:
        cfg: ConfigParser instance.

    Returns:
        WFI client.
    """
    if not cfg:
        cfg = load_config()

    base_url, wfi_headers = init_wfi_headers(cfg)

    return WFIClient(
        base_url=base_url,
        headers=wfi_headers,
    )


class WFIClient:
    """A client to interact with WFI API."""

    FIELDS: "dict[str, dict[str, Any]]" = WFI_FIELDS

    def __init__(
        self,
        base_url: str,
        headers: "dict[str, str]",
        num_retries: int = 5,
    ) -> None:
        """Initialize the WFIClient.

        Args:
            base_url: WFI base url.
            headers: WFI headers containing domain, objectstore, channel, api_key
            num_retries: Number of retries when making requests to WFI.
        """
        self.base_url = base_url

        self.session = requests.Session()
        self.session.headers.update(headers)
        self.session.mount(self.base_url, HTTPAdapter(max_retries=num_retries))

    def checkin_document(
        self,
        files: "dict[str, Any]",
        data: "dict[str, str]",
        document_id: "str | None" = None,
    ) -> str:
        """Check in a document.

        If doc_id is provided, check in as a minor version.

        Args:
            files: A dictionary containing the file-like object.
            data: Document properties.
            document_id: The existing WFI document id.

        Returns:
            The new ID of the checked in document.
        """
        url = f"{self.base_url}/checkindocument"

        # Check in minor version
        if document_id:
            url += f"/{{{document_id}}}/d"

        response = self.session.post(
            url,
            files=files,
            data=data,
        )
        response.raise_for_status()

        new_document_id: str = response.text[1:-1]

        return new_document_id

    def fetch_content(
        self,
        doc_id: str,
    ) -> "dict[Hashable, Any]":
        """Fetch document content from WFI.

        Args:
            doc_id: ID of the document to fetch the content.

        Returns:
            filename, contentBytes and contentType

        Raises:
            ContentDecodingError: If the content is empty.
            InvalidHeader: If the content type or filename is empty.
        """
        url = f"{self.base_url}/fetchcontentbydocid/{{{doc_id}}}"

        res = self.session.get(url)
        res.raise_for_status()

        if not (content := res.content):
            msg = f"Content is empty for document {doc_id}"
            logging.error(msg)
            raise ContentDecodingError(msg)

        if not (content_type := res.headers.get("content-type")):
            msg = f"Content type is empty for document {doc_id}"
            logging.error(msg)
            raise InvalidHeader(msg)

        if not (
            (content_disposition := res.headers.get("content-disposition"))
            and (filename := re.findall("filename=(.+)", content_disposition)[0])
        ):
            msg = f"Filename is empty for document {doc_id}"
            logging.error(msg)
            raise InvalidHeader(msg)
        filename = filename.replace('"', "")

        return {
            "filename": filename,
            "content": content,
            "content-type": content_type,
        }

    def fetch_metadata(
        self,
        doc_id: str,
        fields: "list[str] | None" = None,
    ) -> "dict[str, Any]":
        """Fetch the metadata of a document.

        Args:
            doc_id: ID of the document to fetch the metadata.
            fields: List of fields to fetch.

        Returns:
            A dictionary containing the metadata.
        """
        url = f"{self.base_url}/fetchmetadatabydocid/{{{doc_id}}}"

        if fields:
            response = self.session.post(url, json=fields)
        else:
            response = self.session.get(url)

        response.raise_for_status()

        json_data: dict[str, Any] = response.json()

        return json_data

    def update_metadata(self, doc_id: str, metadata: "dict[str, str]") -> None:
        """Update the metadata of a document.

        Args:
            doc_id: ID of the document to update..
            metadata: Metadata to update
        """
        data: dict[str, list[dict[str, str]]] = {"propertyList": []}

        for field, value in metadata.items():
            if field not in self.FIELDS:
                logging.warning(
                    "Field %s not in valid fields %s",
                    field,
                    list(self.FIELDS),
                )
                continue
            data["propertyList"].append({"value": value, **self.FIELDS[field]["wfi"]})

        if not data["propertyList"]:
            logging.warning("Document %s has no fields to update", doc_id)
            return

        url = f"{self.base_url}/updatemetadata/{{{doc_id}}}"
        response = self.session.put(url, json=data)
        response.raise_for_status()

    def restore_soft_delete(self, doc_id: str) -> None:
        """Restore soft deleted document in WFI.

        Args:
            doc_id: ID of the document to restore.
        """
        data: dict[str, list[dict[str, str | bool]]] = {
            "propertyList": [
                {
                    "value": False,
                    "propertyDataType": BOOLEAN,
                    "propertyName": "IsSoftDeleted",
                },
            ],
        }

        url = f"{self.base_url}/updatemetadata/{{{doc_id}}}"
        response = self.session.put(url, json=data)
        response.raise_for_status()

    def soft_delete(self, doc_id: str) -> None:
        """Soft delete document in WFI.

        Args:
            doc_id: ID of the document to delete.
        """
        data: dict[str, list[dict[str, str | bool]]] = {
            "propertyList": [
                {
                    "value": True,
                    "propertyDataType": BOOLEAN,
                    "propertyName": "IsSoftDeleted",
                },
            ],
        }

        url = f"{self.base_url}/updatemetadata/{{{doc_id}}}"
        response = self.session.put(url, json=data)
        response.raise_for_status()
