"""Redis client."""

import logging
import time
from functools import wraps
from pathlib import Path
from typing import TYPE_CHECKING

from redis import Sentinel, StrictRedis
from redis.sentinel import MasterNotFoundError

from octopus.utils import load_config

if TYPE_CHECKING:
    from collections.abc import Callable
    from typing import Any


REDIS_SERVER_PORT = 6379
REDIS_SENTINEL_SERVER_PORT = 26379


def inject_redis_client(func: "Callable[..., Any]") -> "Callable[..., Any]":
    """Inject Redis client into function arguments.

    Args:
        func: Function to inject Redis client into.

    Returns:
        Wrapped function.
    """

    @wraps(func)
    def wrapper(
        *args: "Any",  # noqa: ANN401
        **kwargs: "Any",  # noqa: ANN401
    ) -> "Any":  # noqa: ANN401, type: ignore[no-any-return]
        if not kwargs.get("redis_client"):
            redis_client = init_redis_client()
            kwargs["redis_client"] = redis_client
        return func(*args, **kwargs)

    return wrapper


def init_redis_client(
    port: int = 6379,
    password: "str | None" = None,
    db: int = 15,
    sentinel_retry_count: int = 3,
    **kwargs: "Any",  # noqa: ANN401
) -> "StrictRedis[bytes]":
    """Initialize Redis client.

    Args:
        port: Redis port. Defaults to 6379.
        password: Redis password. Defaults to None.
        db: Redis database number. Defaults to 15.
        sentinel_retry_count: Number of retries to find the master. Defaults to 3.
        **kwargs: Additional keyword arguments.

    Returns:
        Redis client.

    Raises:
        MasterNotFoundError: If the master is not found after the retries.
    """
    cfg = load_config("/etc/squirro/common.ini")
    host = cfg["redis"].get("host", "localhost")
    password = password or cfg["redis"]["password"]
    for key in list(kwargs.keys()):
        if key.startswith("ssl"):
            kwargs.pop(key)

    sentinel_cfg = Path("/etc/redis/sentinel.conf")
    if sentinel_cfg.exists():
        service_name = (
            "redis-server-master" if port == REDIS_SERVER_PORT else "redis-cache-master"
        )
        logging.info("Connecting to %s via sentinel", service_name)
        hosts = [(h, REDIS_SENTINEL_SERVER_PORT) for h in host.split(",")]
        sentinel = Sentinel(hosts, sentinel_kwargs=kwargs, socket_timeout=5)

        # Retry for cluster nodes as a new master might still be in the process of
        # being elected.
        count = 0
        while count < sentinel_retry_count:
            count += 1
            try:
                client: StrictRedis[bytes] = sentinel.master_for(
                    service_name,
                    redis_class=StrictRedis,
                    password=password,
                    db=db,
                    socket_timeout=5,
                    **kwargs,
                )
            except MasterNotFoundError:
                logging.exception("Master not found. Retrying in 1 second...")
                time.sleep(1)
            else:
                mip: str
                mport: int
                (
                    mip,
                    mport,
                ) = client.get_connection_kwargs()[  # type: ignore[no-untyped-call]
                    "connection_pool"
                ].get_master_address()
                logging.info("Connected to Redis master at %s:%s", mip, mport)
                break
        else:
            msg = f"Unable to find master after {count} retries at {hosts}:{port}."
            logging.error(msg)
            raise MasterNotFoundError(msg)
    else:
        logging.info("Connecting to single Redis instance directly")
        client = StrictRedis(host, port, db, password=password, **kwargs)

    return client
