"""Email client."""

import logging
from dataclasses import dataclass
from email.message import EmailMessage
from pathlib import Path
from smtplib import SMTP

from jinja2 import Template

from octopus.utils import load_config


@dataclass
class SMTPConfig:
    """A class to hold the SMTP configuration."""

    host: str
    login: bool
    password: str
    port: str
    user: str


def init_email_client(*, keep_alive: bool = False) -> "EmailClient":
    """Initialize the email client.

    Args:
        keep_alive: Whether to keep the connection alive. Defaults to False.

    Returns:
        EmailClient: The email client.
    """
    return EmailClient(keep_alive=keep_alive)


def _load_smtp_config() -> SMTPConfig:
    """Load the SMTP configuration from the config file.

    Returns:
        SMTPConfig: The SMTP configuration.

    Raises:
        ValueError: If the username and password are required for login but not
            provided.
    """
    cfg = load_config("/etc/squirro/emailsender.ini")
    host = cfg.get("emailsender", "smtp_server")
    login = cfg.getboolean("emailsender", "smtp_login")
    port = cfg.get("emailsender", "smtp_port")
    user = cfg.get("emailsender", "smtp_username")
    if not user:
        user = "'postmaster@squirro.com'"
    password = cfg.get("emailsender", "smtp_password")
    if login and not (user and password):
        msg = "Username and password are required for login"
        logging.error(msg)
        raise ValueError(msg)

    return SMTPConfig(host, login, password, port, user)


class EmailClient:
    """A class to send emails using the SMTP protocol."""

    connected: bool = False
    smtp: SMTP

    def __init__(self, *, keep_alive: bool = False) -> None:
        """Initialize the email client.

        Args:
            keep_alive: Whether to keep the connection alive. Defaults to False.
        """
        self._keep_alive = keep_alive
        self.cfg = _load_smtp_config()

    def send(
        self,
        *,
        subject: str,
        recipients: "str | list[str]",
        template: str,
        data: "dict[str, dict[str, str]]",
    ) -> None:
        """Send an email to the target recipient.

        Args:
            subject: The subject of the email.
            recipients: The recipient(s) of the email.
            template: The path to email template.
            data:  The data to be rendered in the email template.

        Raises:
            FileNotFoundError: If the template file does not exist.
            ValueError: If the subject, recipients, or template is an empty string.
        """
        if not (subject and recipients and template):
            msg = "Subject, recipients, and template cannot be an empty string."
            logging.error(msg)
            raise ValueError(msg)

        # Convert the recipients to a string
        if isinstance(recipients, list):
            recipients = ", ".join(recipients)

        # Load the email template
        if not (tfp := Path(template)).exists():
            msg = f"Template file not found: {template}"
            logging.error(msg)
            raise FileNotFoundError(msg)

        try:
            with tfp.open(encoding="utf-8") as f:
                t = Template(f.read())
            html = t.render(data)
        except Exception:
            logging.exception("Failed to load the template.")
            raise

        # Create the email message
        email = EmailMessage()
        email["Subject"] = subject
        email["From"] = self.cfg.user
        email["To"] = recipients
        email.set_content(html, subtype="html")

        if not self.connected:
            self._connect()

        self.smtp.sendmail(self.cfg.user, recipients.split(","), email.as_string())

        if not self._keep_alive and self.connected:
            self.disconnect()

    def _connect(self) -> None:
        """Connect to the email server."""
        if self.connected:
            logging.warning("Already connected.")
            return

        try:
            self.smtp = SMTP(self.cfg.host, int(self.cfg.port))
        except Exception:
            logging.exception("Failed to connect to the server.")
            raise

        if self.cfg.login:
            try:
                self.smtp.login(self.cfg.user, self.cfg.password)
            except Exception:
                logging.exception("Failed to login.")
                raise

        self.connected = True
        logging.info("Connected to the SMTP server.")

    def disconnect(self) -> None:
        """Disconnect from the email server."""
        if not self.connected:
            logging.warning("Not connected.")
            return

        try:
            self.smtp.quit()
        except Exception:
            logging.exception("Failed to disconnect.")
            raise

        self.connected = False
        logging.info("Disconnected from the SMTP server.")
