"""Mock WFI plugin."""

import json
import logging
from pathlib import Path
from typing import TYPE_CHECKING
from uuid import uuid4

from flask import jsonify, make_response, request

from octopus.clients import init_redis_client
from octopus.utils import get_mime_type
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from flask import Response


log = logging.getLogger(__name__)
plugin = StudioPlugin(__name__)
redis_client = init_redis_client()
storage_dir = Path("/apps/mock-wfi")
storage_dir.mkdir(exist_ok=True)

MAPPINGS = {
    "created_at": "DateCreated",
    "references": "BBCA_Reference",
    "wfi_company_cif": "BBCA_CIF",
    "wfi_company_name": "BBCA_CustomerName",
    "wfi_company_segment": "BBCA_Segment",
    "wfi_company_rm_code": "BBCA_BusinessUnit",
    "wfi_company_team_name": "BBCA_Team",
    "wfi_document_category": "BBCA_DocumentCategory",
    "wfi_document_type": "BBCA_DocumentType",
    "wfi_document_name": "BBCA_DocumentName",
    "wfi_document_date": "BBCA_DocumentDate",
}


@plugin.route(  # type: ignore[misc]
    "/wfiapi/checkindocument",
    methods=["POST"],
    skip_authentication=True,
)
def check_in_document() -> "Response":
    """Allow check in of document.

    When a document is checked in, we move the document to a separate
    folder and write the mapping to redis.

    Returns:
        The response.
    """
    log.info("Checking in document")
    try:
        props: list[dict[str, str]] = json.loads(request.form.to_dict()["DocProps"])[
            "propertyList"
        ]
    except Exception:
        log.exception("Error parsing DocProps")
        res = make_response(
            jsonify({"status": "failed", "message": "Failed to parse data"})
        )
        res.status_code = 400
        return res

    # Check if the post request has the file part
    if not (f := request.files.get("File")):
        log.error("No file")
        res = make_response(jsonify({"status": "failed", "message": "No file"}))
        res.status_code = 400
        return res

    if not f.filename:
        log.error("No file selected")
        res = make_response(
            jsonify({"status": "failed", "message": "No file selected"})
        )
        res.status_code = 400
        return res

    # Use a random UUID as the document ID
    doc_id = str(uuid4()).upper()

    # Save the file to storage with same extension
    file_path = f"{storage_dir!s}/{doc_id}{Path(f.filename).suffix.lower()}"
    log.info("Saving file %s to %s", f.filename, file_path)
    f.save(file_path)

    # Write the mapping to a redis hash
    metadata = {
        "_filename": f.filename,
        "_path": file_path,
        "content-type": get_mime_type(file_path),
        "Id": doc_id,
    }
    for p in props:
        if (property_name := p["propertyName"]) in MAPPINGS.values():
            metadata[property_name] = p["value"]
    log.info("Writing metadata to redis: %s", metadata)
    redis_client.hset("mock_wfi", doc_id, json.dumps(metadata))

    return make_response(f"{{{doc_id!s}}}")


@plugin.route(  # type: ignore[misc]
    "/wfiapi/fetchcontentbydocid/<doc_id>",
    methods=["GET"],
    skip_authentication=True,
)
def fetch_content_by_doc_id(doc_id: str) -> "Response":
    """Fetch document content by doc_id.

    Args:
        doc_id: The document ID.

    Returns:
        The response.
    """
    log.info("Fetching content for doc_id: `%s`", doc_id)
    data = redis_client.hget("mock_wfi", doc_id[1:-1])
    if not data:
        log.error("Document not found")
        res = make_response(
            jsonify({"status": "failed", "message": "Document not found"})
        )
        res.status_code = 404
        return res

    metadata: dict[str, str] = json.loads(data)
    content = Path(metadata["_path"]).read_bytes()

    # Return response with content, content type and content disposition with filename
    res = make_response(content)
    res.headers["content-type"] = metadata["content-type"]
    res.headers["content-disposition"] = (
        f'attachment; filename="{metadata["_filename"]}"'
    )
    return res


@plugin.route(  # type: ignore[misc]
    "/wfiapi/fetchmetadatabydocid/<doc_id>",
    methods=["GET"],
    skip_authentication=True,
)
def fetch_metadata_by_doc_id(doc_id: str) -> "Response":
    """Fetch metadata by doc_id.

    Args:
        doc_id: The document ID.

    Returns:
        The response.
    """
    log.info("Fetching metadata for doc_id: `%s`", doc_id)
    data = redis_client.hget("mock_wfi", doc_id[1:-1])
    if not data:
        log.error("Document not found")
        res = make_response(
            jsonify({"status": "failed", "message": "Document not found"})
        )
        res.status_code = 404
        return res

    return make_response(
        json.dumps(
            {
                "propertyNameValuePair": {
                    k: v for k, v in json.loads(data).items() if k in MAPPINGS.values()
                }
            }
        ).encode("utf-8")
    )


@plugin.route(  # type: ignore[misc]
    "/wfiapi/updatemetadata/<doc_id>",
    methods=["PUT"],
    skip_authentication=True,
)
def update_metadata(doc_id: str) -> "Response":
    """Update metadata.

    Args:
        doc_id: The document ID.

    Returns:
        The response.
    """
    log.info("Updating metadata for doc_id: `%s`", doc_id)
    return make_response("", 204)
