"""Email sending job."""

import logging
from typing import TYPE_CHECKING

from octopus.clients import init_email_client
from octopus.email import GROUP_NAME, STREAM_NAME, EmailPayload
from octopus.stream import Streamer
from octopus.utils import set_log_verbosity

if TYPE_CHECKING:
    from typing import Any

    from octopus.clients import EmailClient

set_log_verbosity(logging.INFO)


class EmailSendingStreamer(Streamer):
    """Email sending streamer."""

    email_client: "EmailClient"

    def process(self, payload: "dict[str, Any]") -> None:
        """Process message.

        Args:
            payload: Payload to process.
        """
        logging.info("Payload: %s", payload)
        email = EmailPayload.from_dict(payload)

        try:
            subject: str = email.subject
            recipients: str = email.recipients
            template: str = email.template
            self.email_client.send(
                subject=subject,
                recipients=recipients,
                template=template,
                data=email.data,
            )
        except Exception:
            logging.exception("Failed to send email.")
            raise

    def preprocess(self) -> None:
        """Preprocess."""
        self.email_client = init_email_client(keep_alive=True)

    def postprocess(self) -> None:
        """Postprocess."""
        self.email_client.disconnect()


if __name__ == "__main__":
    EmailSendingStreamer(STREAM_NAME, GROUP_NAME).run()
