"""Update the status report index."""

import logging
import sys

# Add integration/studio/document_status_tracking to path
sys.path.append("/opt/squirro/octopus/integration/studio/document_status_tracking")


from sqlalchemy import create_engine
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker
from status_tracking.models import DocumentStatus

from octopus.clients import init_squirro_client
from octopus.utils import load_config, set_log_verbosity

set_log_verbosity(logging.INFO)

Base = declarative_base()


def main() -> None:
    """Entrypoint."""
    db = load_config("/etc/squirro/frontend.ini")["studio"]["db"]
    logging.info("Creating database engine at %s", db)
    engine = create_engine(db)

    logging.info("Initializing Squirro client")
    sq_client, project_id = init_squirro_client()
    logging.info("Querying items")
    res = sq_client.query(
        project_id,
        query=(
            '-current_doc_status_header:"*WFI*"'
            ' AND -current_doc_status_header:"End*"'
            ' AND current_doc_status:"*"'
        ),
        count=10000,
    )
    logging.info("Found %d items", res["count"])

    logging.info("Creating database session")
    session = sessionmaker(bind=engine)()
    try:
        for item in res["items"]:
            logging.info("Checking item `%s`", item["id"])
            doc: DocumentStatus = session.query(DocumentStatus).get(item["id"])
            logging.info(
                "Found document `%s` with current status_trail_report_idx=%d",
                item["id"],
                doc.status_trail_report_idx,
            )
            if not doc:
                logging.warning("Item `%s` not found", item["id"])
                continue
            idx = len(doc.status_trail) - 1
            for status in doc.status_trail[::-1]:
                logging.info("Checking status `%s`", status["code"])
                if not status["code"].startswith("AT"):
                    logging.info("Setting status report index to %d", idx)
                    doc.status_trail_report_idx = idx
                    break
                idx -= 1
            else:
                logging.info("Setting status report index to 1")
                doc.status_trail_report_idx = 1
            session.add(doc)
            session.commit()
            logging.info(
                "Updated status report index for `%s` to %d",
                item["id"],
                doc.status_trail_report_idx,
            )
        logging.info("Committing changes")
    except Exception:
        logging.exception("Error updating status report index")
    finally:
        session.close()


if __name__ == "__main__":
    main()
