"""Retag indexed items."""

import logging
from argparse import ArgumentParser
from typing import TYPE_CHECKING

from octopus.clients import init_squirro_client
from octopus.utils import set_log_verbosity

if TYPE_CHECKING:
    from argparse import Namespace
    from typing import Any

set_log_verbosity(2)


def main(args: "Namespace") -> None:
    """Entrypoint.

    Args:
        args: Parsed arguments.
    """
    sq_client, project_id = init_squirro_client()
    label = args.label
    query = f"{label}:{args.old_value}"

    num_items = 0
    items: list[dict[str, Any]] = []

    for item in sq_client.scan(project_id, query=query, count=10000):
        items.append(
            {
                "id": item["id"],
                "keywords": {
                    label: [args.new_value],
                },
            }
        )
        num_items += 1

        if len(items) > args.batch_size:
            sq_client.modify_items(project_id, items)
            items = []

    if items:
        sq_client.modify_items(project_id, items)

    logging.info(
        "Updated %s from %s to %s for %i items",
        label,
        args.old_value,
        args.new_value,
        num_items,
    )


def init_args() -> "Namespace":
    """Initialize arguments.

    Returns:
        Parsed arguments.
    """
    parser = ArgumentParser()
    parser.add_argument(
        "-l", "--label", help="Squirro label to be updated", required=True
    )
    parser.add_argument(
        "-o", "--old-value", help="Old value to overwrite", required=True
    )
    parser.add_argument(
        "-n", "--new-value", help="New value for the label", required=True
    )
    parser.add_argument(
        "-s",
        "--batch_size",
        type=int,
        help="Number of items to update in a batch",
        default=1000,
    )
    return parser.parse_args()


if __name__ == "__main__":
    args = init_args()
    main(args)
