"""Rename a data source."""

import logging
from argparse import ArgumentParser
from typing import TYPE_CHECKING

from octopus.clients import init_squirro_client
from octopus.utils import set_log_verbosity

set_log_verbosity(logging.INFO)


if TYPE_CHECKING:
    from argparse import Namespace


def main(args: "Namespace") -> None:
    """Entrypoint.

    Args:
        args: Parsed arguments.

    Raises:
        ValueError: If no source is found with the given name or ID or multiple
            sources are found with the given name.
    """
    sq_client, project_id = init_squirro_client()

    source_id: str | None = args.source_id
    name_old: str | None = args.name_old
    name_new: str = args.name_new

    # Need either the id or the current name
    if not (source_id or name_old):
        msg = "Please provide the id or name of the source to update."
        logging.error(msg)
        raise ValueError(msg)

    if source_id:
        source = sq_client.get_source(project_id, source_id, include_config=True)
    else:
        matches = [
            source
            for source in sq_client.get_sources(project_id, include_config=True)
            if source["name"] == name_old
        ]

        if not matches:
            msg = "No source found with the given name."
            logging.error(msg)
            raise ValueError(msg)

        if len(matches) > 1:
            msg = f"Multiple sources found with name {name_old}."
            logging.error(msg)
            raise ValueError(msg)

        source = matches[0]

    config = source["config"]
    config["dataloader_options"]["plugin_name"] = "wfi_loader"

    sq_client.modify_source(
        project_id,
        source["id"],
        name=name_new,
        config=config,
    )


def init_args() -> "Namespace":
    """Initialize arguments.

    Returns:
        Parsed arguments.
    """
    parser = ArgumentParser()
    parser.add_argument("-s", "--source-id", help="ID of the datasource to update")
    parser.add_argument("-o", "--name-old", help="Name of the datasource to update")
    parser.add_argument("-n", "--name-new", help="New name to rename to", required=True)
    return parser.parse_args()


if __name__ == "__main__":
    args = init_args()
    main(args)
