"""Generate batch reports."""

import argparse
import logging
from datetime import datetime, timedelta
from pathlib import Path
from typing import TypedDict

import pytz

from octopus.clients import init_squirro_client
from octopus.data.reports import generate_batch_report
from octopus.utils import set_log_verbosity


class BatchReportOptions(TypedDict):
    """Data required to generate a batch report."""

    report_id: str
    name: str
    current_doc_status: list[str]
    query_options: dict[str, str | None]


set_log_verbosity(logging.INFO)


LOCAL_TIMEZONE = pytz.timezone("Asia/Singapore")
DATETIME_FORMAT = "%Y-%m-%dT%H:%M:%S"

DEFAULT_BBCA_STATUSES = ["001A", "001B", "001E", "001F"]
DEFAULT_CS_STATUSES = ["001", "001Z", "001C", "001E", "002A", "002A1", "003A", "003A1"]


def main(args: "argparse.Namespace") -> None:
    """Entrypoint."""
    sq_client, project_id = init_squirro_client()

    project_config = sq_client.get_project_configuration(project_id)["config"]
    num_outstanding_days: dict[str, int] = project_config.get(
        "octopus.reports.num-outstanding-days", {}
    ).get("value", {})
    statuses: dict[str, list[str]] = project_config.get(
        "octopus.reports.statuses", {}
    ).get("value", {})

    # To consider different timezones if servers are in different locations.
    # Script runs on the following day.
    today = datetime.now(LOCAL_TIMEZONE).replace(
        hour=0, minute=0, second=0, microsecond=0
    )
    report_date = today - timedelta(days=1)
    report_date_utc = report_date.astimezone(pytz.utc)

    outstanding_days_ago = {
        report_id: (report_date_utc - timedelta(days=num_days)).strftime(
            DATETIME_FORMAT
        )
        for report_id, num_days in num_outstanding_days.items()
        if num_days != -1  # -1 to show all outstanding documents, do not check by date
    }

    reports_options: list[BatchReportOptions] = [
        {
            "report_id": "ECO001",
            "name": "Outstanding Documents for BBCA CSO",
            "current_doc_status": statuses.get("ECO001", DEFAULT_BBCA_STATUSES),
            "query_options": {"created_before": outstanding_days_ago.get("ECO001")},
        },
        {
            "report_id": "ECO002",
            "name": "Outstanding documents for BBCA Lodgement Team",
            "current_doc_status": statuses.get("ECO002", DEFAULT_CS_STATUSES),
            "query_options": {"created_before": outstanding_days_ago.get("ECO002")},
        },
        {
            "report_id": "ECO003",
            "name": "Documents Created",
            "current_doc_status": [],
            "query_options": {
                "created_after": report_date_utc.strftime(DATETIME_FORMAT)
            },
        },
    ]

    for report_options in reports_options:
        logging.info("Generating %s", report_options["report_id"])

        report_name = (
            # replace segment later
            f"{report_options['report_id']}_{{segment}}_{report_options['name']}_"
            f"{report_date.strftime('%Y%m%d')}.xlsx"
        )

        generate_batch_report(
            statuses=report_options.get("current_doc_status", []),
            query_options=report_options["query_options"],
            fp=Path(args.output_dir, report_name).as_posix(),
        )


def parse_args() -> "argparse.Namespace":
    """Parse arguments.

    Returns:
        Arguments
    """
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--output-dir",
        help="Output directory where reports are exported to",
        default="",
    )
    return parser.parse_args()


if __name__ == "__main__":
    args = parse_args()
    main(args)
