"""Company data index."""

import json
import logging
from pathlib import Path
from typing import TYPE_CHECKING

import pandas as pd
import requests

from octopus.data import CompanyDataIndex, get_users_info
from octopus.utils import load_config, set_log_verbosity

if TYPE_CHECKING:
    from collections.abc import Hashable

set_log_verbosity(logging.INFO)


COMPANY_DATA_MAPPING = {
    "Country": "country",
    "Cust_Segment_Full": "company_segment_full",
    "CustSIBSCIFKey": "company_cif",
    "CustSIBSName": "company_name",
    "permission_code": "permission_code",
    "RMLanID": "wfi_company_rm_code",
    "RMName": "rm_name",
    "Segment": "wfi_company_segment",
    "Team": "wfi_company_team_name",
    "Team_CD": "wfi_company_team_code",
    "Unique_Identifier": "company_uid",
}


def df2dict(df: pd.DataFrame) -> "list[dict[Hashable, list[str]]]":
    """Convert a pandas dataframe to a company data list.

    Args:
        df: The pandas dataframe to convert.

    Returns:
        A list of dictionaries with the company data.
    """
    # Set all values in row to string array
    df = df.apply(lambda col: col.map(lambda x: [str(x)]))

    # Convert each row to a dictionary with the correct keys as string
    df = df.rename(columns=COMPANY_DATA_MAPPING)
    return df.to_dict(orient="records")


def get_rm_emails_mapping() -> "dict[str, list[str]]":
    """Get the mapping of RM codes to emails.

    Returns:
        A dictionary with the RM codes as keys and the emails as values.
    """
    user_info = get_users_info()
    return {
        rm_code: [email]
        for user in user_info
        if (rm_code := user.get("rmcode")) and (email := user.get("email"))
    }


def main(company_data_file: str) -> None:
    """Entrypoint."""
    path = Path(company_data_file)
    if not path.exists():
        logging.error("File %s does not exist", path)
        return

    cfg = load_config()

    logging.info("Loading company data from %s", path)
    company_data = pd.read_csv(path, dtype=str, encoding="latin-1")
    company_data = company_data.fillna("")

    logging.info("Converting df to list of dictionaries")
    company_data_list = df2dict(company_data)

    # Load RM emails mapping
    logging.info("Tagging RM emails to company data.")
    rm_code_email_mapping = get_rm_emails_mapping()
    for idx, company in enumerate(company_data_list):
        if not (rm_code := company.get("wfi_company_rm_code")):
            continue
        rm_email = rm_code_email_mapping.get(rm_code[0], [])
        if rm_email:
            company_data_list[idx]["rm_email"] = rm_email

    # Load permission code mapping
    if cfg["squirro"]["environment"] != "INTERNAL":
        logging.info("Tagging permission codes to company data.")
        # Load permission code mapping
        if not (
            path := Path("/flash/octopus/cache/permission_code_mapping.json")
        ).exists():
            logging.error("File %s does not exist", path)
            return
        with path.open(encoding="utf-8") as f:
            permission_code_mapping: dict[str, list[str]] = json.load(f)

        for idx, company in enumerate(company_data_list):
            if (team_code := company.get("wfi_company_team_code", [""])[0]) and (
                permission_code := permission_code_mapping.get(team_code, [])
            ):
                company_data_list[idx]["permission_code"] = permission_code

    # Build the index and save it
    logging.info("Building index")
    company_data_index = CompanyDataIndex.build_index(company_data_list)
    company_data_index.save_index()

    # Regenerate the autocomplete index
    requests.get(
        "http://localhost/studio/autocomplete_api/"
        f"regenerate_company_names/{cfg['squirro']['project_id']}"
        f"?token={cfg['squirro']['token']}",
        timeout=600,
    )

    logging.info("Done")


if __name__ == "__main__":
    from argparse import ArgumentParser

    parser = ArgumentParser()
    parser.add_argument(
        "-i",
        "--company_data_file",
        type=str,
        required=True,
        help="Path to the company data file",
    )
    args = parser.parse_args()
    # This needed as the company segment data is inconsistent on UAT.
    try:
        main(args.company_data_file)
    except Exception:
        logging.exception("Unable to generate company data index.")
