import pytest
import sync_company_changes


class TestCompareCompanyData:
    @pytest.mark.parametrize(
        "label_changed, old_value, expected_value",  # noqa: PT006
        [
            ({"company_name": ["Company C"]}, "Company B", "Company C"),
            ({"permission_code": ["C1"]}, "B1", "C1"),
            ({"wfi_company_rm_code": ["R002"]}, "R001", "R002"),
            ({"wfi_company_segment": ["C"]}, "B", "C"),
            ({"wfi_company_team_code": ["C1"]}, "B1", "C1"),
            ({"wfi_company_team_name": ["Team C"]}, "Team B", "Team C"),
        ],
    )
    def test_single_label_change(
        self, label_changed, old_value, expected_value, company_data
    ):
        uid = company_data["company_uid"][0]
        label_name = next(iter(label_changed))

        companies_updated = sync_company_changes.compare_company_data(
            {uid: company_data}, [{**company_data, **label_changed}]
        )

        assert companies_updated[uid][label_name]["old"] == [old_value]
        assert companies_updated[uid][label_name]["new"] == [expected_value]

    def test_multiple_labels_change(self, company_data):
        uid = company_data["company_uid"][0]

        companies_updated = sync_company_changes.compare_company_data(
            {uid: company_data},
            [
                {
                    **company_data,
                    "wfi_company_rm_code": ["R002"],
                    "wfi_company_segment": ["C"],
                    "wfi_company_team_code": ["C1"],
                    "wfi_company_team_name": ["Team C"],
                },
            ],
        )

        assert dict(companies_updated) == {
            "SG_11111": {
                "wfi_company_rm_code": {"old": ["R001"], "new": ["R002"]},
                "wfi_company_segment": {"old": ["B"], "new": ["C"]},
                "wfi_company_team_code": {"old": ["B1"], "new": ["C1"]},
                "wfi_company_team_name": {"old": ["Team B"], "new": ["Team C"]},
            }
        }


class TestUpdateItems:
    def test_update_single_item(self, mock_squirro_client, items):
        mock_squirro_client.query.return_value = {
            "count": 1,
            "eof": True,
            "items": items[:1],
        }

        sync_company_changes.update_items(
            {"SG_11111": {"wfi_company_rm_code": {"old": ["R001"], "new": ["R002"]}}}
        )

        mock_squirro_client.modify_items.assert_called_once_with(
            "project_id",
            [{"id": "id-1", "keywords": {"wfi_company_rm_code": ["R002"]}}],
        )

    def test_update_multiple_items(self, mock_squirro_client, items):
        mock_squirro_client.query.return_value = {
            "count": 2,
            "eof": True,
            "items": items,
        }

        sync_company_changes.update_items(
            {"SG_11111": {"wfi_company_rm_code": {"old": ["R001"], "new": ["R002"]}}}
        )

        mock_squirro_client.modify_items.assert_called_once_with(
            "project_id",
            [
                {"id": "id-1", "keywords": {"wfi_company_rm_code": ["R002"]}},
                {"id": "id-2", "keywords": {"wfi_company_rm_code": ["R002"]}},
            ],
        )

    def test_update_items_company_name_change(self, mock_squirro_client, items):
        mock_squirro_client.query.return_value = {
            "count": 1,
            "eof": True,
            "items": items[:1],
        }

        sync_company_changes.update_items(
            {"SG_11111": {"company_name": {"old": ["Company B"], "new": ["Company C"]}}}
        )

        mock_squirro_client.modify_items.assert_called_once_with(
            "project_id",
            [
                {
                    "id": "id-1",
                    "keywords": {"company_name": ["Company A", "Company C"]},
                }
            ],
        )

    def test_update_items_multiple_company_names_change(
        self, mock_squirro_client, items
    ):
        mock_squirro_client.query.return_value = {
            "count": 1,
            "eof": True,
            "items": items[:1],
        }

        sync_company_changes.update_items(
            {
                "SG_11111": {
                    "company_name": {"old": ["Company B"], "new": ["Company C"]}
                },
                "SG_22222": {
                    "company_name": {"old": ["Company A"], "new": ["Company D"]}
                },
            }
        )

        mock_squirro_client.modify_items.assert_called_once_with(
            "project_id",
            [
                {
                    "id": "id-1",
                    "keywords": {"company_name": ["Company D", "Company C"]},
                }
            ],
        )

    def test_update_items_permission_change(self, mock_squirro_client, items):
        mock_squirro_client.query.return_value = {
            "count": 1,
            "eof": True,
            "items": items[:1],
        }

        sync_company_changes.update_items(
            {"SG_11111": {"permission_code": {"old": ["B1"], "new": ["C1"]}}}
        )

        args, _ = mock_squirro_client.modify_items.call_args
        assert set(args[1][0]["keywords"]["permission_code"]) == {"A1", "C1"}

    def test_update_items_multiple_companies_permission_change(
        self, mock_squirro_client, items
    ):
        mock_squirro_client.query.return_value = {
            "count": 1,
            "eof": True,
            "items": items[:1],
        }

        sync_company_changes.update_items(
            {
                "SG_11111": {"permission_code": {"old": ["B1"], "new": ["C1"]}},
                "SG_22222": {"permission_code": {"old": ["A1"], "new": ["D1"]}},
            }
        )

        args, _ = mock_squirro_client.modify_items.call_args
        assert set(args[1][0]["keywords"]["permission_code"]) == {"C1", "D1"}

    def test_update_items_query_twice(self, mock_squirro_client, items):
        mock_squirro_client.query.side_effect = [
            {
                "count": 1,
                "eof": False,
                "next_params": {"start": 1},
                "items": items[:1],
            },
            {
                "count": 1,
                "eof": True,
                "items": items[1:],
            },
        ]

        sync_company_changes.update_items(
            {"SG_11111": {"wfi_company_rm_code": {"old": ["R001"], "new": ["R002"]}}}
        )

        mock_squirro_client.modify_items.assert_called_once_with(
            "project_id",
            [
                {
                    "id": "id-1",
                    "keywords": {"wfi_company_rm_code": ["R002"]},
                },
                {
                    "id": "id-2",
                    "keywords": {"wfi_company_rm_code": ["R002"]},
                },
            ],
        )

    def test_update_items_two_batches(self, mock_squirro_client, items):
        num_items = 1555
        mock_squirro_client.query.return_value = {
            "count": num_items,
            "eof": True,
            "items": [{**items[0], "id": f"id-{i}"} for i in range(num_items)],
        }

        sync_company_changes.update_items(
            {"SG_11111": {"wfi_company_rm_code": {"old": ["R001"], "new": ["R002"]}}}
        )

        assert mock_squirro_client.modify_items.call_count == 2
        first_call = mock_squirro_client.modify_items.call_args_list[0][0]
        assert len(first_call[1]) == 1000
        second_call = mock_squirro_client.modify_items.call_args_list[1][0]
        assert len(second_call[1]) == 555


@pytest.fixture
def company_data():
    return {
        "company_cif": ["11111"],
        "company_name": ["Company B"],
        "company_uid": ["SG_11111"],
        "permission_code": ["B1"],
        "wfi_company_rm_code": ["R001"],
        "wfi_company_segment": ["B"],
        "wfi_company_team_code": ["B1"],
        "wfi_company_team_name": ["Team B"],
    }


@pytest.fixture
def items():
    return [
        {
            "id": "id-1",
            "keywords": {
                "company_name": ["Company A", "Company B"],
                "uid_permission_code": ["SG_22222___A1", "SG_11111___B1"],
            },
        },
        {
            "id": "id-2",
            "keywords": {
                "company_name": ["Company A", "Company B"],
                "uid_permission_code": ["SG_22222___A1", "SG_11111___B1"],
            },
        },
    ]
