"""Organisation structure."""

import logging
from collections import defaultdict
from typing import TYPE_CHECKING

import pandas as pd

if TYPE_CHECKING:
    from pathlib import Path


class TerritoryNode:
    """Class to represent a node in the territory tree."""

    def __init__(self, nid: str, name: str, node_type: str) -> None:
        """Initialize the node."""
        self.nid = nid
        self.name = name
        self.node_type = node_type
        self.parent: TerritoryNode | None = None
        self.children: list[TerritoryNode] = []

    def add_child(self, node: "TerritoryNode") -> None:
        """Add a child to the node."""
        self.children.append(node)
        node.parent = self

    def __str__(self, level: "int" = 0) -> "str":
        """String representation of the node.

        Args:
            level: The level of the node in the tree.

        Returns:
            A string representation of the node.
        """
        ret: str = "\t" * level + repr(self.nid) + "\n"
        for child in self.children:
            ret += child.__str__(level + 1)
        return ret


def construct_mappings(file_path_csv: "Path") -> "dict[str, list[str]]":
    """Construct mappings from the CSV file.

    Args:
        file_path_csv: Path to the CSV file.

    Returns:
        A dictionary mapping team codes to their parent IDs.

    Raises:
        FileNotFoundError: If the file does not exist.
    """
    if not file_path_csv.exists():
        msg = f"Invalid file type or file not found - {file_path_csv}"
        logging.error(msg)
        raise FileNotFoundError(msg)

    df: pd.DataFrame = pd.read_csv(file_path_csv, encoding="latin-1")

    all_nodes: dict[str, TerritoryNode] = {}
    team_code_nodes: set[TerritoryNode] = set()
    team_codes: defaultdict[str, list[str]] = defaultdict(list)

    # Create a tree-like data structure
    for _, row in df.iterrows():
        child_id = row["ChildID"]
        child_name = row["ChildName"]
        child_type = row["ChildType"]
        parent_id = row["ParentID"]
        parent_name = row["ParentName"]
        parent_type = row["ParentType"]

        child_node: TerritoryNode | None = all_nodes.get(child_id)
        if child_node is None:
            child_node = TerritoryNode(child_id, child_name, child_type)
            all_nodes[child_id] = child_node

        parent_node: TerritoryNode | None = all_nodes.get(parent_id)
        if parent_node is None:
            parent_node = TerritoryNode(parent_id, parent_name, parent_type)
            all_nodes[parent_id] = parent_node

        parent_node.add_child(child_node)

        if child_type == "Team":
            team_code_nodes.add(child_node)

    # team_code -> all the parents of team_code
    for team_code_node in team_code_nodes:
        current: TerritoryNode | None = team_code_node
        while current:
            team_codes[team_code_node.nid].append(current.nid)
            current = current.parent

    return dict(team_codes)
