# mypy: ignore-errors
"""Studio plugin to create and manage user roles information with metadata,
that is relevant for query templating and entitlement handling.

This studio plugin enables storing of user information in the format - {
user@ocbc.com:{         "ecm_role": "admin",         "job_title":
"WCM(WCM 80) RM",         "org_unit_id": "RE078",
"org_unit_type": "Team",         "rmcode": "myecm3",         "role":
"RM",         "role_ocbc": "Workbench:RM",     } }

where users' emails are mapped to the relevant user information
"""

import json
import logging
from typing import TYPE_CHECKING

import requests
from flask import jsonify, request

from octopus.clients.redis_client import init_redis_client
from octopus.utils import load_config
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:

    from flask import Response


plugin = StudioPlugin(__name__)
log = logging.getLogger(__name__)

ERROR_ADDING_USER_ROLES = "Exception occured while adding user roles"
USER_ROLES_DICT_NAME = "user_roles_hash"

ERROR_GETTING_USER_ROLES = "Exception occured when trying to retrieve user roles"
ERROR_DELETING_USER_ROLES = "Exception occured while deleting user roles"
FILE_PATH_INI: str = "/opt/squirro/octopus/config/main.ini"

VALID_ROLES = ["admin", "user", "reader"]


redis_client = init_redis_client()
cfg = load_config()


@plugin.route("/user_roles", skip_authentication=True)
# pylint: disable-next=too-many-return-statements
def get_all_users() -> "Response":
    """View a specific or all users currently stored in Squirro's Redis DB
    This route handles both -
        1. Searching for a particular user's details
        2. Looking up the entire DB for all users
    """
    users = []
    emails = request.args.getlist("email")

    # if user email is passed as a URL param to search user information
    if len(emails) != 0:
        for email in emails:
            if email == "":
                continue

            try:
                single_user_role_info: bytes | None = redis_client.hget(
                    USER_ROLES_DICT_NAME, email.lower()
                )
            except Exception:  # pylint: disable=broad-except
                return response_json(
                    "failure",
                    "Exception occured while looking up user information",
                    503,
                )

            if single_user_role_info is None:
                return response_json(
                    "failure",
                    f"No user found with email {email}",
                    400,
                )

            try:
                decoded_user_info: str = single_user_role_info.decode("utf-8")
                json_user_info: dict[str, str] = json.loads(decoded_user_info)
            except Exception:  # pylint: disable=broad-except
                return response_json(
                    "failure",
                    "Exception occured while parsing user information",
                    503,
                )

            users.append(json_user_info)

        return jsonify({"users": users})

    try:
        user_roles_info_raw: dict[bytes, bytes] = redis_client.hgetall(
            USER_ROLES_DICT_NAME
        )
        user_roles_info: dict[str, str] = {
            key.decode(): value.decode() for key, value in user_roles_info_raw.items()
        }
    except Exception:  # pylint: disable=broad-except
        return response_json(
            "failure",
            "Exception occured when trying to retrieve user roles",
            503,
        )

    for value in user_roles_info.values():
        try:
            value_dict: dict[str, str] = json.loads(value)
        except Exception:  # pylint: disable=broad-except
            return response_json(
                "failure",
                "Exception occured when trying to parse user roles info",
                503,
            )

        users.append(value_dict)

    return jsonify({"users": users})


@plugin.route("/user_roles", methods=["DELETE"], skip_authentication=True)
def delete_user_info() -> "tuple[str, int] | str":
    """Delete users from Squirro's Redis DB."""
    emails = request.args.getlist("email")
    if len(emails) == 0:
        return response_json(
            "failure",
            "No email specified. Please pass in an email in the query parameter",
            400,
        )

    try:
        for email in emails:
            email_lower = email.lower()

            if redis_client.hget(USER_ROLES_DICT_NAME, email_lower) is None:
                return response_json(
                    "failure",
                    f"Invalid email address {email} - email address does not exist",
                    400,
                )

            redis_client.hdel(USER_ROLES_DICT_NAME, email_lower)

    except Exception:  # pylint: disable=broad-except
        return response_json(
            "failure",
            ERROR_DELETING_USER_ROLES,
            503,
        )

    # Regenerate the autocomplete index
    requests.get(
        "http://localhost/studio/autocomplete_api/"
        f"regenerate_users/{cfg['squirro']['project_id']}"
        f"?token={cfg['squirro']['token']}",
        timeout=10,
    )

    return response_json(
        "success",
        f"Deleted {len(emails)} user(s)",
        200,
    )


@plugin.route("/user_roles", methods=["POST"], skip_authentication=True)
# pylint: disable-next=too-many-return-statements
def add_user_info() -> "Response":
    """Add users and their user roles in Squirro's Redis DB."""
    content_type: str = request.headers.get("Content-Type", "")
    if content_type != "application/json":
        return response_json(
            "failure",
            "Invalid request data format passed - only application/json is accepted",
            415,
        )

    try:
        request_data: dict = json.loads(request.data)
    except Exception:  # pylint: disable=broad-except
        return response_json(
            "failure",
            "Exception occured when parsing request body",
            400,
        )

    users: dict[str, str] | list = request_data.get("users", [])
    if not isinstance(users, list) or len(users) == 0:
        return response_json(
            "failure",
            "No users found in the request body",
            503,
        )

    try:
        for user_info in users:
            is_valid, message = validate_user_info(user_info=user_info)
            if not is_valid:
                return response_json(
                    "failure",
                    message,
                    400,
                )

            email = user_info["email"]
            email_lower = email.lower()

            if redis_client.hget(USER_ROLES_DICT_NAME, email_lower) is not None:
                return response_json(
                    "failure",
                    f"Email address already exists - {email} already exists "
                    "in the DB. Use PATCH method to modify existing user's information",
                    400,
                )

            redis_client.hset(
                USER_ROLES_DICT_NAME,
                email_lower,
                json.dumps(user_info).encode("utf-8"),
            )

    except Exception:  # pylint: disable=broad-except
        return response_json(
            "failure",
            ERROR_ADDING_USER_ROLES,
            503,
        )

    # Regenerate the autocomplete index
    requests.get(
        "http://localhost/studio/autocomplete_api/"
        f"regenerate_users/{cfg['squirro']['project_id']}"
        f"?token={cfg['squirro']['token']}",
        timeout=10,
    )

    return response_json(
        "success",
        f"Created {len(users)} user(s)",
        201,
    )


@plugin.route("/user_roles", methods=["PATCH"], skip_authentication=True)
# pylint: disable-next=too-many-return-statements
def modify_user_info() -> "Response":
    """Modify users and their user roles in Squirro's Redis DB.

    Any one field or multiple fields of multiple users can be sent for
    modification.
    """
    content_type: str = request.headers.get("Content-Type", "")
    if content_type != "application/json":
        return response_json(
            "failure",
            "Invalid request data format passed - only application/json is accepted",
            415,
        )

    try:
        request_data: dict = json.loads(request.data)
    except Exception:  # pylint: disable=broad-except
        return response_json(
            "failure",
            "Exception occured when parsing request body",
            400,
        )

    users: dict[str, str] | list = request_data.get("users", [])
    if not isinstance(users, list) or len(users) == 0:
        return response_json(
            "failure",
            "No users found in the request body",
            503,
        )

    try:
        for user_info in users:
            is_valid, message = validate_user_info(
                user_info=user_info, validate_org_unit_id=False
            )
            if not is_valid:
                return response_json(
                    "failure",
                    message,
                    400,
                )

            email = user_info["email"]
            email_lower = email.lower()

            existing_user_info_raw: bytes | None = redis_client.hget(
                USER_ROLES_DICT_NAME, email_lower
            )
            if existing_user_info_raw is None:
                return response_json(
                    "failure",
                    f"Invalid email address {email} - "
                    "email address does not exist. Add user via the POST method",
                    400,
                )

            existing_user_info: dict[str, str] = json.loads(
                existing_user_info_raw.decode()
            )
            existing_user_info.update(user_info)

            redis_client.hset(
                USER_ROLES_DICT_NAME,
                email_lower,
                json.dumps(existing_user_info).encode("utf-8"),
            )
    except Exception:  # pylint: disable=broad-except
        return response_json(
            "failure",
            ERROR_ADDING_USER_ROLES,
            503,
        )

    # Regenerate the autocomplete index
    requests.get(
        "http://localhost/studio/autocomplete_api/"
        f"regenerate_users/{cfg['squirro']['project_id']}"
        f"?token={cfg['squirro']['token']}",
        timeout=10,
    )

    return response_json(
        "success",
        f"Modified {len(users)} user(s)",
        200,
    )


def validate_user_info(
    user_info: "dict[str,str]", validate_org_unit_id: bool = True
) -> "tuple[bool, str]":
    """Validates user information being passed. The fields are checked and the
    relevant status code with error/success message is returned.

    Parameters:
    user_info (Dict[str, str]): The user info dictionary to validate
    validate_org_unit_id (bool): Whether to validate org_unit_id or not

    Returns:
    Tuple[bool, str]: A tuple containing a boolean indicating validity and a message.
    """
    email = user_info.get("email")
    if email is None:
        return False, "Missing email for one or more users"

    # validate role
    ecm_role: str = user_info.get("ecm_role", "")
    if ecm_role not in VALID_ROLES:
        return False, f"Invalid user role - {ecm_role} for {email}"

    if validate_org_unit_id and user_info.get("org_unit_id") is None:
        return False, f"Missing org_unit_id for {email}"

    return True, "Valid user info"


def response_json(status, msg, status_code):
    if status == "failure":
        log.exception(msg)
    else:
        log.info(msg)

    res = jsonify({"status": status, "message": msg})
    res.status_code = status_code
    return res
