"""Storage plugin."""

import logging
from io import BytesIO
from typing import TYPE_CHECKING

from flask import make_response, send_file

from octopus.clients.wfi_client import init_wfi_client
from octopus.utils import set_log_verbosity
from squirro.common.config import get_config
from squirro.common.dependency import get_injected
from squirro.lib.storage.handler import StorageHandler
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from flask import Response


set_log_verbosity(logging.INFO)

plugin = StudioPlugin(__name__)
storage = StorageHandler(get_config("squirro.lib.storage"))
wfi_client = init_wfi_client()


@plugin.route(  # type: ignore[misc]
    "/wfi/<project_id>/<wfi_document_id>", methods=["GET"], allow_project_readers=True
)
def wfi(project_id: str, wfi_document_id: str) -> "Response":
    """WFI endpoint.

    Args:
        project_id: The project ID.
        wfi_document_id: The WFI document ID.

    Returns:
        The WFI document.
    """
    title = get_document_title(project_id, wfi_document_id)

    # Return error response if the user is not entitled to see the document
    if not title:
        return make_response("You are not entitled to view the document", 403)

    # Fetch the content of the given item from storage and return it to the user
    try:
        data: bytes = storage.download(f"storage://wfi/{project_id}/{wfi_document_id}")
    except Exception:
        logging.exception(
            "Exception occurred when downloading file with WFI Document ID %s",
            wfi_document_id,
        )
        raise

    res: Response = send_file(
        BytesIO(data),
        mimetype="application/octet-stream",
    )
    return res


@plugin.route(  # type: ignore[misc]
    "/wfi/download/<project_id>/<wfi_document_id>",
    methods=["GET"],
    allow_project_readers=True,
)
def wfi_download(project_id: str, wfi_document_id: str) -> "Response":
    """WFI download endpoint.

    Args:
        project_id: The project ID.
        wfi_document_id: The WFI document ID.

    Returns:
        The WFI document.
    """
    title = get_document_title(project_id, wfi_document_id)

    # Return error response if the user is not entitled to see the document
    if not title:
        return make_response("You are not entitled to view the document", 403)

    content = wfi_client.fetch_content(wfi_document_id)

    if not isinstance(content_type := content["content-type"], str):
        return make_response("Invalid content type", 400)

    if content_type == "application/pdf" and not title.endswith(".pdf"):
        title += ".pdf"

    res: Response = send_file(
        BytesIO(content["content"]),
        mimetype=content_type,
        as_attachment=True,
        download_name=title,
    )
    return res


def get_document_title(project_id: str, wfi_document_id: str) -> str | None:
    """Get the document title.

    Args:
        project_id: The project ID.
        wfi_document_id: The WFI document ID.

    Returns:
        The document title.
    """
    squirro_client = get_injected("squirro_client")

    try:
        res = squirro_client.query(
            project_id, f'wfi_document_id:"{wfi_document_id}"', count=1
        )
    except Exception:
        logging.exception(
            "Failed to query document with WFI Document ID `%s`", wfi_document_id
        )
        raise

    if not res.get("count"):
        return None

    title: str = res["items"][0]["title"]
    return title
