"""Status permissions plugin."""

import json
import logging
from collections import defaultdict
from typing import TYPE_CHECKING

from flask import jsonify, make_response, request

from octopus.clients import init_redis_client
from squirro.common.dependency import get_injected
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from typing import Any

    from flask import Response

    RoleBasedTableType = dict[str, dict[str, dict[str, dict[str, dict[str, bool]]]]]


log = logging.getLogger(__name__)
plugin = StudioPlugin(__name__)


def _load_status_permissions_config() -> "dict[str, Any]":
    redis_client = init_redis_client()
    status_permissions: dict[str, Any] = {}
    if data := redis_client.get("status_permissions"):
        status_permissions = json.loads(data.decode())
    return status_permissions


def generate_role_based_table(config: "dict[str, Any]") -> "RoleBasedTableType":
    """Generate role based table.

    Args:
        config: The config.

    Returns:
        The role based table.
    """
    if not config:
        return {}

    ops_roles = [
        "Workbench:CSSupportChecker",
        "Workbench:CSSupportMaker",
        "Workbench:CSSupportMaker&Checker",
        "iLMS:BBCACSOChecker",
        "iLMS:BBCACSOMaker",
        "iLMS:BLTOfficer",
    ]
    role_based_mapping: RoleBasedTableType = {}
    for role in ops_roles:
        role_based_mapping[role] = defaultdict(
            lambda: defaultdict(
                lambda: defaultdict(
                    lambda: {"delete": False, "edit": False, "restore": False}
                )
            )
        )

    for current_status, previous_statuses in config.items():  # noqa: PLR1702
        for previous_status, uploaders in previous_statuses.items():
            for uploader, actions in uploaders.items():
                for k, v in actions.items():
                    if isinstance(v, bool):
                        for role in ops_roles:
                            role_based_mapping[role][current_status][previous_status][
                                uploader
                            ][k] = v
                        continue
                    for role in v:
                        role_based_mapping[role][current_status][previous_status][
                            uploader
                        ][k] = True

    return role_based_mapping


role_based_mapping = generate_role_based_table(_load_status_permissions_config())


@plugin.route(  # type: ignore[misc]
    "/<project_id>/permissions/<user_role>", methods=["GET"], allow_project_readers=True
)
def permissions(project_id: str, user_role: str) -> "Response":
    """Get role based mapping.

    Args:
        project_id: The project ID.
        user_role: The user role.

    Returns:
        The role based mapping.
    """
    log.debug("Getting role based mapping")
    return make_response(
        role_based_mapping.get(user_role, {}),
        200,
    )


@plugin.route(  # type: ignore[misc]
    "/get_config", methods=["GET"], allow_project_readers=True
)
def get_config() -> "Response":
    """Get status permissions config.

    Returns:
        The status permissions config.
    """
    log.debug("Getting status permissions config")
    return make_response(
        {
            "status_permissions": _load_status_permissions_config(),
        },
        200,
    )


@plugin.route("/set_config", methods=["POST"])  # type: ignore[misc]
def set_config() -> "Response":
    """Set status permissions config.

    Returns:
        The status permissions config.
    """
    log.debug("Setting status permissions config")
    data: dict[str, Any] | None = request.get_json()
    if not data:
        return make_response("No data provided", 400)

    try:
        config = data["status_permissions"]
    except Exception:
        return make_response("Key `status_permissions` does not exist.", 400)

    redis_client = init_redis_client()

    try:
        redis_client.set("status_permissions", json.dumps(config))
    except Exception:
        log.exception("Failed to update status permissions configs")
        return make_response(
            "Failed to update status permissions configs. Please try again.", 500
        )

    global role_based_mapping  # noqa: PLW0603
    role_based_mapping = generate_role_based_table(config)

    return make_response(jsonify(config), 200)


@plugin.route(  # type: ignore[misc]
    "/<project_id>/default_statuses/<user_role>", allow_project_readers=True
)
def get_default_statuses(project_id: str, user_role: str) -> "Response":
    """Get default statuses.

    Args:
        project_id: The project ID.
        user_role: The user role.

    Returns:
        The default statuses.
    """
    config_handler = get_injected("config_handler")
    roles_default_statuses = config_handler.get_project_value(
        "octopus.roles.default-statuses", project_id=project_id
    )
    return jsonify(roles_default_statuses.get(user_role, []))


@plugin.route(  # type: ignore[misc]
    "/<project_id>/viewable_roles/<user_role>", allow_project_readers=True
)
def get_viewable_roles(project_id: str, user_role: str) -> "Response":
    """Get viewable roles.

    Args:
        project_id: The project ID.
        user_role: The user role.

    Returns:
        The viewable roles.
    """
    config_handler = get_injected("config_handler")
    viewable_roles = config_handler.get_project_value(
        "octopus.roles.viewable-roles", project_id=project_id
    )
    return jsonify(viewable_roles.get(user_role, []))
