"""Logs plugin."""

import logging
from io import BytesIO
from pathlib import Path
from typing import TYPE_CHECKING

from flask import send_file

from octopus.utils import load_config
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from flask import Response


log = logging.getLogger(__name__)
plugin = StudioPlugin(__name__)
url = load_config()["squirro"]["url"]


def download(name: str, *, error: bool = False) -> "Response":
    """Download logs endpoint.

    Args:
        name: Which logs to download
        error: Whether to download the error log

    Returns:
        The log.
    """
    with Path(f"/var/log/squirro/{name}/{'stderr' if error else name}.log").open(
        "rb"
    ) as f:
        log_bio = BytesIO(f.read())

    return send_file(  # type: ignore[no-any-return]
        log_bio,
        as_attachment=True,
        mimetype="text/plain",
        download_name=f"{'stderr' if error else name}.log",
    )


@plugin.route("/<name>", skip_authentication=True)  # type: ignore[misc]
def download_log(name: str) -> "Response":
    """Logs endpoint.

    Args:
        name: Which log to return

    Returns:
        The log.
    """
    return download(name)


@plugin.route("/error/<name>", skip_authentication=True)  # type: ignore[misc]
def download_log_error(name: str) -> "Response":
    """Error logs endpoint.

    Args:
        name: Which error log to return

    Returns:
        The log.
    """
    return download(name, error=True)
