"""Logout plugin."""

import logging
from typing import TYPE_CHECKING

from flask import make_response, render_template, session

from octopus.clients import init_squirro_client
from octopus.utils import load_config
from squirro.common.dependency import get_injected
from squirro.integration.frontend.context import execute_in_studioaware_context
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from flask import Response

    from squirro_client import SquirroClient


log = logging.getLogger(__name__)
plugin = StudioPlugin(__name__)
url = load_config()["squirro"]["url"]


@execute_in_studioaware_context  # type: ignore[misc]
def _logout_get_user_details() -> str:
    """Get user ID and logout.

    This is a workaround for OCBC that is based off the platform code as their
    SSO in combination with custom logout page prevents session from being
    cleared properly.

    Returns:
        The user ID.
    """
    user_id: str = session.get("user_id", "")
    session_id = session.get("session_id")
    if user_id and session_id:
        try:
            client, _ = init_squirro_client()
            client.delete_session(user_id, session_id)
            get_injected("user_session_cache").clear(additional_prefix=session_id)
        except Exception:
            log.exception("Could not delete the session with ID %s", session_id)
    session.clear()

    return user_id


@plugin.route("/", skip_authentication=True)  # type: ignore[misc]
def logout() -> "Response":
    """Logout endpoint.

    Returns:
        The logout response.
    """
    user_id = _logout_get_user_details()
    lan_id = "Unknown"
    country = ""
    try:
        sq_client: SquirroClient = get_injected("squirro_client")
        lan_id = sq_client.get_user_data(user_id)["user_information"]["lan_id"][0]
        country = "SG"
    except Exception:  # pylint: disable=broad-except
        logging.exception("Failed to get user data")

    res = make_response(
        render_template("index.html", url=url, user_id=lan_id, country=country)
    )
    res.delete_cookie("session")  # Clear user session
    return res
