# mypy: ignore-errors
import base64
import json
import logging
import os
import os.path
import pprint
import tempfile
from typing import TYPE_CHECKING
from urllib.parse import urlparse

import pandas as pd
from flask import Response, jsonify, make_response, request
from itsdangerous import URLSafeSerializer
from saml2 import BINDING_HTTP_POST, BINDING_HTTP_REDIRECT, entity
from saml2.client import Saml2Client
from saml2.config import Config as Saml2Config
from saml2.metadata import create_metadata_string
from saml2.s_utils import UnravelError, decode_base64_and_inflate
from saml2.saml import NAMEID_FORMAT_EMAILADDRESS, NameID

from octopus.clients import init_redis_client
from octopus.utils import load_config
from squirro.common.dependency import get_injected
from squirro.sdk.studio import StudioPlugin
from squirro_client import ItemUploader

if TYPE_CHECKING:
    from typing import Dict, List, Optional

plugin = StudioPlugin(__name__)
log = logging.getLogger(__name__)

CALLBACK_URL = "/sso/callback"
ACCEPTED_TIME_DIFF = 60
SERVICE_NAME = "squirro"
DEFAULT_ENTITY_ID = "https://sso.squirro.com/o/saml2/entity"
VALID_ROLES = {"admin", "user", "demo", "reader", "reject"}

# https://github.com/IdentityPython/pysaml2/pull/548 removed the handling of
# friendly names. So we need to use the OID instead.
FIELDS_GIVEN_NAME = ["urn:oid:2.5.4.42", "givenName", "firstName"]
FIELDS_SURNAME = ["urn:oid:2.5.4.4", "surname"]
FIELD_ROLE = "Role"

FILE_PATH_INI: "str" = "/opt/squirro/octopus/config/main.ini"
RECON_DASHBOARD_USERS_FILE_PATH: "str" = "/flash/octopus/ReconDashboardUsers.csv"
USER_ROLES_DICT_NAME: "str" = "user_roles_hash"

cache_files: "Dict" = {}

cfg = load_config()

uploader = ItemUploader(
    project_id=cfg["activity"]["project_id"],
    cluster=cfg["squirro"]["cluster"],
    token=cfg["squirro"]["token"],
    source_name="Login Stats",
    pipeline_workflow_name="Standard",
)

role_to_group = {
    "iLMS:BBCACSOMaker": "Ops",
    "iLMS:BBCACSOChecker": "Ops",
    "iLMS:BLTOfficer": "Ops",
    "Workbench:CSSupportMaker": "Ops",
    "Workbench:CSSupportChecker": "Ops",
    "Workbench:CSSupportMaker&Checker": "Ops",
}


def retrieve_recon_dashboard_users(csv_path: str, col_name: str) -> "List[str]":
    try:
        df = pd.read_csv(csv_path)
    except Exception:
        log.exception("Failed to read %s", csv_path)
        raise

    try:
        email_col = df[col_name].str.lower()
    except Exception:
        log.exception("Failed to retrieve %s from %s", col_name, csv_path)
        raise

    return email_col.tolist()


class InvalidRoleError(Exception):
    """InvalidRoleError."""


class InvalidConfigError(Exception):
    """InvalidConfigError."""


def get_url_serializer():
    """Return a `URLSafeSerializer` object used to safely transmit data.

    This is used to pass the `RelayState` to the SSO provider, and then
    validate it when we get it back.
    """
    config = get_injected("config")
    encrypt_key = config.get("security", "url_serialization_token")
    return URLSafeSerializer(encrypt_key)


def get_saml_client(config):
    """Given the name of an IdP, return a configuation.

    The configuration is a hash for use by saml2.config.Config
    """
    metadata_file = get_cache_filename(config["id"], "metadata")
    if not os.path.exists(metadata_file):
        with open(metadata_file, "w", encoding="utf-8") as f:
            f.write(config["metadata"])

    cert_file = None
    if config.get("certificate"):
        cert_file = get_cache_filename(config["id"], "certificate")
        if not os.path.exists(cert_file):
            with open(cert_file, "w", encoding="utf-8") as f:
                f.write(config["certificate"])

    acs_url = config["fqdn"].rstrip("/") + CALLBACK_URL
    https_acs_url = "/sso/callback"
    settings = {
        "entityid": config.get("entity_id"),
        "accepted_time_diff": ACCEPTED_TIME_DIFF,
        "metadata": {"local": [metadata_file]},
        "service": {
            "sp": {
                "name": SERVICE_NAME,
                "endpoints": {
                    "assertion_consumer_service": [
                        (acs_url, BINDING_HTTP_REDIRECT),
                        (acs_url, BINDING_HTTP_POST),
                        (https_acs_url, BINDING_HTTP_REDIRECT),
                        (https_acs_url, BINDING_HTTP_POST),
                    ],
                    "single_logout_service": [
                        (acs_url, BINDING_HTTP_REDIRECT),
                        (acs_url, BINDING_HTTP_POST),
                    ],
                },
                # Don't verify that the incoming requests originate from us via
                # the built-in cache for authn request ids in pysaml2
                "allow_unsolicited": True,
                # Don't sign authn requests, since signed requests only make
                # sense in a situation where you control both the SP and IdP
                "authn_requests_signed": False,
                "logout_requests_signed": False,
                "want_assertions_signed": True,
                "want_response_signed": False,
            }
        },
    }
    if cert_file:
        settings["cert_file"] = cert_file

    spconfig = Saml2Config()
    spconfig.load(settings)
    spconfig.allow_unknown_attributes = True
    saml_client = Saml2Client(config=spconfig)
    return saml_client


def error_response(msg: str, code: int) -> Response:
    res = jsonify({"error": msg})
    res.status_code = code
    return res


@plugin.route("/extauth", methods=["POST"], skip_authentication=True)
def extauth():
    data = json.loads(request.data)
    log.debug("extauth request received: %s", pprint.pformat(data))

    config = get_saml_config(domain=data.get("domain"))
    if not config.get("enabled"):
        log.debug("extauth_saml not enabled: %r", config)
        ret = make_response("")
        ret.status_code = 204
        return ret

    if not config.get("entity_id"):
        # Use the default entity_id
        config["entity_id"] = DEFAULT_ENTITY_ID

    saml_client = get_saml_client(config)

    if data.get("action") == "logout":
        # User is logging out - send them to the SLO
        return _initiate_logout(saml_client, config, data)
    if data.get("action") == "login":
        return _initiate_login(saml_client, data)
    if data.get("action") == "callback":
        return _process_callback(saml_client, data, config)
    raise ValueError("Unknown action")


def _initiate_login(saml_client, data):
    # Prepare authentication and redirect to identity provider
    relay_state = get_url_serializer().dumps(data.get("url"), salt="saml_relay")
    _, info = saml_client.prepare_for_authenticate(
        nameid_format="urn:oasis:names:tc:SAML:1.1:nameid-format:unspecified",
        relay_state=relay_state,
    )

    redirect_url = None
    for key, value in info["headers"]:
        if key == "Location":
            redirect_url = value
    return jsonify({"redirect": redirect_url})


def _initiate_logout(saml_client, _, data):
    binding = BINDING_HTTP_REDIRECT
    entity_ids = saml_client.config.metadata.keys()
    entity_id = entity_ids[0]
    srvs = saml_client.metadata.single_logout_service(entity_id, binding, "idpsso")
    if not srvs:
        log.debug("No SLO service has been configured")
        return jsonify({"logout": True})

    user_info = data["user_info"]
    request = saml_client.do_logout(
        NameID(format=NAMEID_FORMAT_EMAILADDRESS, text=user_info["email"]),
        entity_ids=[entity_id],
        reason="User logged out",
        expire=None,
        expected_binding=binding,
    )

    for _, logout_info in request.items():
        if isinstance(logout_info, tuple):
            req_binding, http_info = logout_info
            if req_binding == BINDING_HTTP_REDIRECT:
                for key, value in http_info["headers"]:
                    if key.lower() == "location":
                        return jsonify({"redirect": value})

                res = jsonify({"error": "Missing Location header"})
                res.status_code = 500
            else:
                res = jsonify({"error": f"Unknown logout binding: {req_binding}"})
                res.status_code = 500
            return res

    # Default return
    return jsonify({"logout": True})


def _process_callback(saml_client, data, config):
    message = get_key_from_request(data, "SAMLResponse")
    try:
        xmlstr = decode_base64_and_inflate(message)
    except Exception:  # pylint: disable=broad-except
        xmlstr = base64.b64decode(message)

    is_logout = b"LogoutResponse" in xmlstr.split(b">", 1)[0]

    if is_logout:
        return _process_logout_callback(saml_client, data, message)
    return _process_acs_callback(saml_client, data, message, config)


# pylint: disable=too-many-branches,too-many-locals,too-many-statements
def _process_acs_callback(saml_client, data, message, config):
    # Validate callback and get info
    authn_response = saml_client.parse_authn_request_response(
        message, entity.BINDING_HTTP_POST
    )
    authn_response.get_identity()
    user_info = authn_response.get_subject()
    username: str = user_info.text
    redirect_to = None
    relay_state = get_key_from_request(data, "RelayState")
    if relay_state:
        try:
            url = get_url_serializer().loads(relay_state, salt="saml_relay")
            redirect_to = validate_redirect_url(url, config["fqdn"])
        except ValueError as e:
            log.warning("Invalid relay state %r: %s", relay_state, e)
            res = jsonify({"status": "error", "error": "Invalid redirect URL."})
            res.status_code = 400
            return res

    group_ids = []
    if config.get("group_id"):
        group_ids.append(config.get("group_id"))

    redis_client = init_redis_client()
    user_info_redis: "Optional[bytes]" = redis_client.hget(
        USER_ROLES_DICT_NAME, username.lower()
    )

    if user_info_redis is None:
        return error_response("User not found in redis", 404)

    user_info_dict: "Dict[str, str]" = json.loads(user_info_redis)
    user_role: "Optional[str]" = user_info_dict.get("ecm_role")
    if user_role is None or user_role not in ["admin", "user", "reader"]:
        return error_response("Invalid user role", 400)

    # Since normal users are saved as "users" in redis, assignment to reader is
    # necessary to prevent Setup space access
    if user_role == "user":
        user_role = "reader"

    user_group_name = role_to_group.get(user_info_dict["role_ocbc"], "RM")
    group_names = [user_group_name]

    recon_dashboard_users = retrieve_recon_dashboard_users(
        RECON_DASHBOARD_USERS_FILE_PATH, "Value.email"
    )
    if username.lower() in recon_dashboard_users:
        group_names.append("recon_dashboard_users")

    ret = {
        "tenant": config.get("tenant"),
        "user_id": username,
        "email": username,
        "group_ids": group_ids,
        "redirect": redirect_to,
        "role": user_role,
        "group_names": group_names,
    }

    if config.get("user_data_fields") or config.get("session_data_fields"):
        ret["user_data"] = get_custom_attributes_from_response(
            "user", config, authn_response.ava
        )
        ret["session_data"] = get_custom_attributes_from_response(
            "session", config, authn_response.ava
        )
    else:
        # If neither `user_data` or `session_data` is specified, fallback to
        # the old behaviour of saving everything on the session
        ret["user_information"] = authn_response.ava
        # Populate with additional information from redis
        for key, val in user_info_dict.items():
            if key == "role_ocbc" and val == "Workbench:CSSupportMaker&Checker":
                ret["user_information"][key] = [
                    "Workbench:CSSupportMaker",
                    "Workbench:CSSupportChecker",
                ]
                continue
            ret["user_information"][key] = [val]

    # Check if we got the name
    full_name = []
    full_name.extend(get_response_value(authn_response.ava, FIELDS_GIVEN_NAME))
    full_name.extend(get_response_value(authn_response.ava, FIELDS_SURNAME))
    full_name = list(filter(None, full_name))
    if full_name:
        ret["fullname"] = " ".join(full_name)

    # Return groups based on the provided roles
    if config.get("groups_field") and authn_response.ava.get(config["groups_field"]):
        ret["group_names"] = authn_response.ava[config["groups_field"]]

    if config.get("groups_roles"):
        role = get_user_role(ret.get("group_names"), config.get("groups_roles"))
        if role == "reject":
            log.warning("User access denied based on role configuration.")
            return error_response("Permission denied to User", 403)
        ret["role"] = role

    uploader.upload(
        [
            {
                "id": get_injected("short_uuid"),
                "title": f"Login {username}",
                "keywords": {
                    "action": ["user.login"],
                    "user_email": [username],
                    "user_name": [" ".join(full_name)],
                    "project_title": ["OCtopus"],
                    **{key: [val] for key, val in user_info_dict.items()},
                },
            }
        ]
    )

    return jsonify(ret)


def _process_logout_callback(saml_client, _, message):
    try:
        saml_client.parse_logout_request_response(message, BINDING_HTTP_REDIRECT)
    except UnravelError:
        saml_client.parse_logout_request_response(message, BINDING_HTTP_POST)
    return jsonify({"logout": True, "redirect": None})


def get_saml_settings(config):
    metadata_file = get_cache_filename(config["id"], "metadata")
    if not os.path.exists(metadata_file):
        with open(metadata_file, "w", encoding="utf-8") as f:
            f.write(config["metadata"])
    cert_file = None
    if config.get("certificate"):
        cert_file = get_cache_filename(config["id"], "certificate")
        if not os.path.exists(cert_file):
            with open(cert_file, "w", encoding="utf-8") as f:
                f.write(config["certificate"])
    acs_url = config["fqdn"].rstrip("/") + CALLBACK_URL
    https_acs_url = "/sso/callback"
    settings = {
        "entityid": config.get("entity_id"),
        "accepted_time_diff": ACCEPTED_TIME_DIFF,
        "metadata": {"local": [metadata_file]},
        "service": {
            "sp": {
                "name": SERVICE_NAME,
                "endpoints": {
                    "assertion_consumer_service": [
                        (acs_url, BINDING_HTTP_REDIRECT),
                        (acs_url, BINDING_HTTP_POST),
                        (https_acs_url, BINDING_HTTP_REDIRECT),
                        (https_acs_url, BINDING_HTTP_POST),
                    ],
                    "single_logout_service": [
                        (acs_url, BINDING_HTTP_REDIRECT),
                        (acs_url, BINDING_HTTP_POST),
                    ],
                },
                # Don't verify that the incoming requests originate from us via
                # the built-in cache for authn request ids in pysaml2
                "allow_unsolicited": True,
                # Don't sign authn requests, since signed requests only make
                # sense in a situation where you control both the SP and IdP
                "authn_requests_signed": False,
                "logout_requests_signed": False,
                "want_assertions_signed": True,
                "want_response_signed": False,
            }
        },
        "allow_unknown_attributes": True,
    }

    if cert_file:
        settings["cert_file"] = cert_file
    return settings


@plugin.route("/save", methods=["POST"])
# pylint: disable-next=too-many-branches, too-many-statements
def save():
    client = get_injected("squirro_client")
    data = json.loads(request.data)
    log.debug("Storing extauth_saml config: %r", request.data)

    redis_client = get_injected("redis_studio")
    if data.get("id"):
        config = json.loads(redis_client.get(f"extauth_saml:{data['id']}"))
    else:
        config = {}
        config["id"] = get_injected("short_uuid")

    try:
        parse_roles(data.get("groups_roles"))
    except InvalidRoleError as e:
        log.warning("Invalid role returned: %s", e)
        res = jsonify({"status": "error", "error": str(e)})
        res.status_code = 400
        return res

    config["groups_roles"] = data.get("groups_roles")
    config["domain"] = data.get("domain")
    config["enabled"] = data.get("enabled")
    config["group_id"] = data.get("group_id")
    config["groups_field"] = data.get("groups_field")
    config["entity_id"] = data.get("entity_id")
    config["tenant"] = client.tenant
    if not config.get("domain"):
        config["domain"] = "*"
    if config["domain"] == "*":
        config["fqdn"] = "https://" + request.headers["Host"].split(":")[0]
    else:
        config["fqdn"] = "https://" + config["domain"]

    if not config.get("entity_id"):
        # Use the default entity_id
        config["entity_id"] = DEFAULT_ENTITY_ID

    if data.get("metadata"):
        metadata_file = data["metadata"]
        path = metadata_file["path"]
        metadata = client.get_tempfile(path)
        # We get a byte-string here, but need a string as we'll put it into JSON.
        metadata = metadata.decode("utf8")
        config["metadata"] = metadata

    if data.get("certificate"):
        cert_file = data["certificate"]
        path = cert_file["path"]
        certificate = client.get_tempfile(path)
        # We get a byte-string here, but need a string as we'll put it into JSON.
        certificate = certificate.decode("utf8")
        config["certificate"] = certificate

    try:
        if "user_data" in data:
            config.update(parse_custom_attributes_config("user", data["user_data"]))
        if "session_data" in data:
            config.update(
                parse_custom_attributes_config("session", data["session_data"])
            )
    except InvalidConfigError as e:
        log.exception("Invalid attributes configuration provided.")
        res = jsonify({"status": "error", "error": str(e)})
        res.status_code = 400
        return res

    redis_client = get_injected("redis_studio")
    redis_client.set(f"extauth_saml:{config['id']}", json.dumps(config))
    log.debug("Saving extauth_saml config - full config is: %s", config)

    # Remove cached files
    cache_files = [
        get_cache_filename(config["id"], "metadata"),
        get_cache_filename(config["id"], "certificate"),
    ]
    for fname in cache_files:
        if os.path.exists(fname):
            os.unlink(fname)

        # Using rmdir instead of shutil.rmtree to avoid erroneous mass
        # deletion.
        # Ignoring any errors, as the worst case scenario here is an empty - or
        # at the very worst a near-empty - folder in the `/tmp` folder which is
        # usually cleaned up by the OS automatically.
        cache_dirname = os.path.dirname(fname)
        try:
            os.rmdir(cache_dirname)
        except Exception:  # pylint: disable=broad-except
            log.exception(
                "Ignoring error on cache directory cleanup of %s", cache_dirname
            )

    return jsonify(config)


@plugin.route("/configs", methods=["GET"])
def configs():
    """Return all configured setups."""
    configs = get_extauth_configs()
    ret = []
    for value in sorted(configs.values(), key=lambda c: c["id"]):
        if "metadata" in value:
            # Not outputting metadata in list
            del value["metadata"]
        ret.append(value)
    return jsonify(ret)


@plugin.route("/configs/<config_id>", methods=["DELETE"])
def config(config_id):
    redis_client = get_injected("redis_studio")
    redis_client.delete(f"extauth_saml:{config_id}")

    ret = make_response("")
    ret.status_code = 204
    return ret


@plugin.route("/groups", methods=["GET"])
def groups():
    """Return list of groups, used for the groups drop-down."""
    client = get_injected("squirro_client")
    groups = client.get_groups()
    options = [{"id": "", "title": "(No group)"}]

    for group in groups:
        options.append({"id": group["id"], "title": group["name"]})

    return jsonify({"options": options})


@plugin.route("/metadata", methods=["GET"], skip_authentication=True)
def get_sp_metadata():
    config = get_saml_config(domain=None)
    if not config.get("enabled"):
        ret = make_response("SAML not enabled")
        ret.status_code = 204
        return ret

    if not config.et("entity_id"):
        # Use the default entity_id
        config["entity_id"] = DEFAULT_ENTITY_ID

    settings = get_saml_settings(config)

    # workaround to now show idp certificate in sp metadata
    if "cert_file" in settings:
        del settings["cert_file"]

    spconfig = Saml2Config()
    spconfig.load(settings, metadata_construction=True)

    sp_metadata = create_metadata_string(
        "", config=spconfig, valid=False, cert=False, sign=False
    )

    ret = make_response(sp_metadata)
    ret.headers["Content-type"] = "text/xml; charset=utf-8"
    return ret


def get_saml_config(domain):
    configs = get_extauth_configs()
    if configs.get(domain):
        config = configs[domain]
    elif configs.get("*"):
        config = configs["*"]
    else:
        return {}

    # Check if all the required config is there.
    if not all([config.get("tenant"), config.get("metadata"), config.get("fqdn")]):
        log.warning("Disabling SSO due to missing tenant, metadata or fqdn: %r", config)
        config["enabled"] = False

    return config


def get_extauth_configs():
    redis_client = get_injected("redis_studio")
    keys = redis_client.keys("extauth_saml:*")
    configs = {}
    if keys:
        for v in redis_client.mget(keys):
            value = json.loads(v)
            configs[value["domain"]] = value
    return configs


def get_cache_filename(config_id, file_type):
    """Return a filename where `file_type` cache for `config_id` is stored.

    This creates a secure random file name on the first time it's
    called.
    """
    key = (config_id, file_type)
    tmpdir = cache_files.get(key)
    if not tmpdir or not os.path.isdir(tmpdir):
        tmpdir = tempfile.mkdtemp()
        cache_files[key] = tmpdir
    tmpfile = os.path.join(tmpdir, file_type)
    return tmpfile


def parse_roles(roles_config):
    """Returns a sorted list of group to role assignments.

    This is used by `get_user_role` to determine the right role for a user's
    group memberships.

    This is done using a simple semicolon-separated expression, where each
    element can have the form `[group_name=]role_name`.

    Example expression which gives `admin` role to squirro ops group, `user` to
    squirro employees, and everybody else is set to `reader`:

        squirro ops=admin; squirro employees=user; reader

    A special role `reject` can be assigned, to disallow logins. This can be
    used in circumstances where the original system doesn't limit logins to the
    system, but login should be restricted to certain groups. Example:

        NA-APP-Squirro-Restricted=user; NA-APP-Squirro_Users=user; reject
    """
    ret: "List" = []
    if not roles_config or not roles_config.strip():
        return ret

    for expression in roles_config.split(";"):
        expression = expression.strip()
        if "=" in expression:
            group_name, role = expression.split("=")
        else:
            group_name = "*"
            role = expression
        group_name = group_name.strip()
        role = role.strip().lower()

        # Ensure the role is valid
        if role not in VALID_ROLES:
            raise InvalidRoleError(f"Role expression uses invalid role {role}.")

        ret.append((group_name, role))

    return ret


def get_user_role(groups, roles_config):
    """Returns the most appropriate role for this user.

    Parses the role configuration expression (see `parse_roles` for
    information). The first expression that matches is used. Matches are done
    in a case insensitive manner.

    Reject is the default "everybody else" role. As a result, if the role
    expression has been defined, then only explicitly allowed groups can access
    the system.
    """
    group_names = []
    if groups:
        for group in groups:
            group_names.append(group.lower())

    parsed_roles = parse_roles(roles_config)
    for group_name, role in parsed_roles:
        if group_name == "*" or group_name.lower() in group_names:
            return role

    return "reject"


def parse_custom_attributes_config(prefix, data):
    """Convert the provided custom attributes config to an internal parsed
    representation."""
    if not data.strip():
        return {}

    raw_lines = []
    parsed_lines = []
    for line in data.split("\n"):
        expression = line.strip()
        if not expression:
            continue

        raw_lines.append(expression)
        if "=" in expression:
            attribute_name, claim = expression.split("=")
        else:
            attribute_name = claim = expression
        attribute_name = attribute_name.strip()
        claim = claim.strip()

        if not attribute_name:
            raise InvalidConfigError(
                f"Missing attribute name in field mapping {expression}"
            )
        if len(attribute_name) > 50:
            raise InvalidConfigError(
                f"Attribute name {attribute_name} too long (maximum 50 characters)"
            )
        if not claim:
            raise InvalidConfigError(f"Missing claim in field mapping {expression}")

        parsed_lines.append({"attribute": attribute_name, "claim": claim})

    return {
        f"{prefix}_data": "\n".join(raw_lines),
        f"{prefix}_data_parsed": parsed_lines,
    }


def get_custom_attributes_from_response(prefix, config, ava):
    ret = {}
    fields = config.get(f"{prefix}_data_parsed", [])
    for field in fields:
        value = get_response_value(ava, [field["claim"]])
        ret[field["attribute"]] = value
    return ret


def get_response_value(ava, keys):
    """Helper to return a given value from the AVA response.

    For each key in the provided keys, this will match case-insensitive
    on response keys ending with the key. The first successful match is
    returned.

    Returns a list of values, because that is how the AVA response
    returns values.
    """
    for key in keys:
        key = key.lower()
        for ava_key, ava_value in ava.items():
            if ava_key.lower().endswith(key):
                return ava_value
    return []


def get_key_from_request(data, key):
    """Depending upon the type of callback, a particular key might exist either
    in form or params.

    We try both and return the first match.
    """
    value = data.get("form", {}).get(key) or data.get("params", {}).get(key)
    return value


def validate_redirect_url(url, fqdn):
    """Ensures that this URL is allowed for redirection.

    This is used for RelayState handling. Raises a `ValueError` for invalid
    URLs.
    """
    if not url:
        return None

    try:
        parsed_url = urlparse(url)
    except ValueError as e:
        raise ValueError("Invalid URL") from e

    domain = parsed_url.hostname
    if parsed_url.scheme and domain != urlparse(fqdn).hostname:
        raise ValueError("Redirect to unknown hostname")

    return url
