"""Email whitelisting plugin."""

# mypy: ignore-errors
import logging
import re
import sys
from typing import TYPE_CHECKING

from flask import jsonify, request
from werkzeug.exceptions import BadRequest

from squirro.common.dependency import get_injected
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from flask import Response

log = logging.getLogger(__name__)
plugin = StudioPlugin(__name__)

# pytest only works with skip_authentication=True
# use skip_authentication=True if using pytest
# otherwise set allow_project_readers=True
plugin_options: "dict[str, bool]" = (
    {"skip_authentication": True}
    if "pytest" in sys.modules
    else {"allow_project_readers": True}
)


@plugin.route("/<project_id>", **plugin_options)
def whitelisted_emails(project_id: str) -> "Response":
    """Get the whitelisted emails for a project.

    Args:
        project_id: The project ID.

    Returns:
        The whitelisted emails.
    """
    client = get_injected("squirro_client")
    config: dict = client.get_project_configuration(project_id=project_id).get(
        "config", {}
    )
    whitelisted_emails: list[str] = (
        config.get("app.whitelisted-emails", {})
        .get("value", {})
        .get("email_addresses", [])
    )

    return jsonify(whitelisted_emails)


@plugin.route("/add_email", methods=["POST"], **plugin_options)
def add_email() -> "tuple[Response, int]":
    """Add an email address to the whitelist.

    Returns:
        The response.

    Raises:
        BadRequest: If the request is missing the project ID or email address or
            if the email address is invalid.
    """
    try:
        json_data = request.json
    except Exception:
        log.exception("Missing parameters in the request.")
        raise

    project_id: str | None = json_data.get("project_id")
    new_email_address: str | None = json_data.get("email_address")

    if project_id is None:
        msg = "Missing project id"
        log.error(msg)
        raise BadRequest(msg)

    if new_email_address is None:
        msg = "Missing email address"
        log.error(msg)
        raise BadRequest(msg)

    email_pattern: str = r"\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,7}\b"
    if not re.match(email_pattern, new_email_address):
        msg = "Invalid email address"
        log.error(msg)
        raise BadRequest(msg)

    config_key: str = "app.whitelisted-emails"

    client = get_injected("squirro_client")
    config: dict = client.get_project_configuration(project_id=project_id).get(
        "config", {}
    )
    whitelisted_emails: list[str] = (
        config.get(config_key, {}).get("value", {}).get("email_addresses", [])
    )

    whitelisted_emails.append(new_email_address)

    client.set_project_configuration(
        project_id=project_id,
        key=config_key,
        value={"email_addresses": whitelisted_emails},
    )

    return jsonify(success=True), 201


@plugin.route("/<project_id>/<email_deleted>", methods=["DELETE"], **plugin_options)
def delete_email(project_id: str, email_deleted: str) -> "tuple[Response, int]":
    """Delete an email address from the whitelist.

    Args:
        project_id: The project ID.
        email_deleted: The email address to delete.

    Returns:
        The response.
    """
    client = get_injected("squirro_client")
    config: dict = client.get_project_configuration(project_id=project_id).get(
        "config", {}
    )
    whitelisted_emails: list[str] = (
        config.get("app.whitelisted-emails", {})
        .get("value", {})
        .get("email_addresses", [])
    )

    config_key: str = "app.whitelisted-emails"

    whitelisted_emails = [
        email for email in whitelisted_emails if email != email_deleted
    ]

    client.set_project_configuration(
        project_id=project_id,
        key=config_key,
        value={"email_addresses": whitelisted_emails},
    )

    return jsonify(success=True), 204
