"""Email config plugin."""

import json
import logging
from typing import TYPE_CHECKING

from flask import make_response, request

from octopus.clients import init_redis_client
from octopus.email import load_email_config
from octopus.utils import load_config, set_log_verbosity
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from flask import Response


set_log_verbosity(logging.INFO)

plugin = StudioPlugin(__name__)
cfg = load_config()
email_config = load_email_config().to_dict()


@plugin.route(  # type: ignore[misc]
    "/get_config", methods=["GET"], allow_project_readers=True
)
def get_config() -> "Response":
    """Get the email config.

    Returns:
        The email config.
    """
    logging.debug("Getting email config")
    return make_response(email_config, 200)


@plugin.route("/set_config", methods=["POST"])  # type: ignore[misc]
def set_config() -> "Response":
    """Set the email config.

    Returns:
        The email config.
    """
    logging.debug("Setting email config")
    data: dict[str, dict[str, str]] | None = request.get_json()
    if not data:
        return make_response("No data provided", 400)

    redis_client = init_redis_client()
    try:
        redis_client.hset(
            "email_notifications",
            "distribution_list",
            json.dumps(data["distribution_list"]),
        )
        redis_client.hset(
            "email_notifications", "triggers", json.dumps(data["triggers"])
        )
    except Exception:
        logging.exception("Failed to update email configs")
        return make_response("Failed to update email configs. Please try again.", 500)

    global email_config  # noqa: PLW0603
    email_config = load_email_config().to_dict()

    return make_response(email_config, 200)
