import pytest
from jsonschema.exceptions import ValidationError

from document_status_tracking.status_tracking import schemas
from document_status_tracking.status_tracking.document_types import DocumentTypes
from document_status_tracking.status_tracking.errors import (
    AccessNotAllowed,
    InvalidRequest,
)
from document_status_tracking.status_tracking.models import (
    DocumentStatus,
    StatusTrackingConfig,
)
from document_status_tracking.status_tracking.util import validate_json


def make_user_info(roles=[], id="non-existing"):
    return {"uid": [id], "lan_id": [id], "role_ocbc": roles}


class TestStatusConfig:
    def setup_method(self) -> None:
        DocumentStatus.document_types = DocumentTypes(
            {
                "ASSIGNMENT": {"bbca_documents": True},
                "CM": {"bbca_documents": False},
            }
        )
        status_data = {
            "access_roles": [
                "Workbench:CSSupportChecker",
                "Workbench:CSSupportMaker",
                "iLMS:BBCACSOChecker",
                "iLMS:BBCACSOMaker",
            ],
            "statuses": [
                {"code": "001"},
                {"code": "001Z"},
                {"code": "001A"},
                {"code": "001B"},
                {"code": "002A"},
            ],
            "transitions": {
                "001": {
                    "001": {"Workbench:CSSupportMaker": ["BBCA", "non-BBCA"]},
                    "001A": {"Workbench:CSSupportChecker": ["BBCA"]},
                    "001Z": {
                        "Workbench:CSSupportMaker": ["BBCA"],
                        "Workbench:CSSupportChecker": ["non-BBCA"],
                    },
                    "001B": {
                        "Workbench:CSSupportChecker": ["BBCA"],
                        "iLMS:BBCACSOChecker": ["BBCA"],
                    },
                    "002A": {"iLMS:BBCACSOChecker": ["BBCA"]},
                }
            },
        }

        self.config = StatusTrackingConfig(
            project_id="test",
            version=1,
            status_map=status_data,
        )
        self.config._init()  # must call manually, not ORM loaded

        self.checker = make_user_info(
            ["Workbench:CSSupportChecker"],
            "checker@e.co",
        )
        self.maker = make_user_info(
            ["Workbench:CSSupportMaker"],
            "maker@e.co",
        )
        self.bbca_checker = make_user_info(
            ["iLMS:BBCACSOChecker"],
            "bbcachecker@e.co",
        )
        self.dual_role = make_user_info(
            ["Workbench:CSSupportChecker", "Workbench:CSSupportMaker"],
            "maker-checker@e.co",
        )

    def make_doc(
        self, status, is_bbca=True, role=["Workbench:CSSupportChecker"], id="j@e.co"
    ):
        return DocumentStatus(
            document_id="document_id",
            project_id="project_id",
            status_trail=[
                {"code": status, "user": {"id": id, "lan_id": id, "role": role}}
            ],
            status_code=status,
            company_name="company_name",
            document_type="ASSIGNMENT" if is_bbca else "CM",
            document_date="document_date",
        )

    def test_followup_status(self) -> None:
        def assert_status(statuses, expected_codes):
            actual = {s["code"] for s in statuses}
            assert actual == set(expected_codes)

        f = self.config.list_available_followup_statuses

        bbca_doc = self.make_doc("001", True, ["Workbench:CSSupportMaker"])
        other_doc = self.make_doc("001", False, ["Workbench:CSSupportMaker"])

        assert_status(f(self.maker, bbca_doc), ["001", "001Z"])
        assert_status(f(self.maker, other_doc), ["001"])
        assert_status(f(self.checker, bbca_doc), ["001A", "001B"])
        assert_status(f(self.checker, other_doc), ["001Z"])
        assert_status(f(self.bbca_checker, bbca_doc), ["002A", "001B"])
        assert_status(f(self.bbca_checker, other_doc), [])

        # Dual role cannot update 001 to 001Z if they uploaded the doc
        doc = self.make_doc(
            "001", True, self.dual_role["role_ocbc"], self.dual_role["lan_id"][0]
        )
        assert_status(f(self.dual_role, doc), ["001", "001A", "001B"])

        # Dual role can update 001 to 001Z if they did not upload the doc
        assert_status(f(self.dual_role, other_doc), ["001", "001Z"])

    def test_update_status_for_dual_role(self) -> None:
        doc = self.make_doc(
            "001", True, self.dual_role["role_ocbc"], self.dual_role["lan_id"][0]
        )
        t = self.config.is_status_update_allowed

        assert t(doc, "001B", self.dual_role) is True
        assert t(doc, "001Z", self.dual_role) is False
        assert t(doc, "001", self.dual_role) is True, "only allowed to edit existing"

        doc = self.make_doc("001", True, ["Workbench:CSSupportMaker"])
        t = self.config.is_status_update_allowed
        assert t(doc, "001B", self.dual_role) is True
        assert t(doc, "001Z", self.dual_role) is True
        assert t(doc, "001", self.dual_role) is True

    def test_update_status(self) -> None:
        bbca_doc = self.make_doc("001", True, ["Workbench:CSSupportMaker"])
        other_doc = self.make_doc("001", False, ["Workbench:CSSupportMaker"])

        t = self.config.is_status_update_allowed
        assert t(bbca_doc, "001Z", self.maker) is True
        assert t(other_doc, "001Z", self.maker) is False
        assert t(bbca_doc, "001Z", self.checker) is False
        assert t(other_doc, "001Z", self.checker) is True
        assert t(bbca_doc, "001Z", self.bbca_checker) is False
        assert t(other_doc, "001Z", self.bbca_checker) is False
        assert t(other_doc, "001", self.bbca_checker) is False
        assert t(other_doc, "001", self.maker) is True
        assert t(bbca_doc, "001", self.maker) is True
        assert t(other_doc, "001", self.checker) is False
        assert t(bbca_doc, "001", self.checker) is False

    def test_access_permission(self) -> None:
        assert (
            self.config.raise_not_allowed_to_access_status_tracking(
                {"role_ocbc": ["Workbench:CSSupportMaker"]}
            )
            is None
        )
        with pytest.raises(AccessNotAllowed):
            self.config.raise_not_allowed_to_access_status_tracking(
                {"role_ocbc": ["RM"]}
            )


class TestDocumentStatus:
    def setup_method(self) -> None:
        DocumentStatus.document_types = DocumentTypes(
            {"ASSIGNMENT": {"bbca_documents": True}}
        )
        self.uploader = {
            "id": "jd@example.com",
            "lan_id": "A001",
            "name": "John Doe",
            "role": ["Workbench:CSSupportMaker"],
        }
        self.config = lambda: None
        statuses = {
            "001": {"header": "001", "description": "001"},
            "005": {"header": "005", "description": "005"},
            "007": {"header": "007", "description": "007"},
            "907": {"header": "907", "description": "907"},
            "906": {"header": "906", "description": "906"},
            "AT001": {
                "description": "From <> to <>",
                "header": "Client name changed",
            },
            "AT002": {
                "description": "From <> to <>",
                "header": "Document type changed",
            },
            "AT003": {
                "description": "From <> to <>",
                "header": "Document date changed",
            },
        }
        self.config.codes = statuses

    def test_create(self) -> None:
        classified = DocumentStatus.create(
            self.config,
            self.uploader,
            "User Upload",
            "document_id",
            "document_name",
            "project_id",
            "company_name",
            "ASSIGNMENT",
            "2024-01-19",
        )
        assert len(classified.status_trail) == 2
        assert classified.status_code == "001"
        assert classified.status_trail[1]["header"] == "001"
        assert classified.is_bbca() is True

        unclassified = DocumentStatus.create(
            self.config,
            self.uploader,
            "User Upload",
            "document_id",
            "document_name",
            "project_id",
        )
        assert unclassified.status_code == "907"

    def test_create_completed(self) -> None:
        completed = DocumentStatus.create(
            self.config,
            self.uploader,
            "WFI Migration",
            "document_id",
            "document_name",
            "project_id",
        )
        assert len(completed.status_trail) == 1
        assert completed.status_code == "007"

    def test_update(self) -> None:
        doc = DocumentStatus.create(
            self.config,
            self.uploader,
            "User Upload",
            "document_id",
            "document_name",
            "project_id",
            "company_name",
            "ASSIGNMENT",
            "2024-01-19",
        )
        assert len(doc.status_trail) == 2
        doc.update(
            self.config,
            self.uploader,
            {"company_name": "ACME Inc.", "document_date": "2000-12-31"},
        )
        assert (
            len(doc.status_trail) == 4
        ), "a new status record created for each updated field"
        assert (
            doc.status_code == doc.status_trail[1]["code"]
        ), "ATxxx should not affect status flow - status code"
        updated_desc = doc.status_trail[-2]["description"]
        print(updated_desc)
        assert (
            "19/01/2024" in updated_desc and "31/12/2000" in updated_desc
        ), "description should contain old and new values"

        with pytest.raises(InvalidRequest):
            doc.update(
                self.config,
                self.uploader,
                {"company_name": "ACME Inc."},  # identical to current value
            )

    def test_auto_classify(self) -> None:
        doc = DocumentStatus.create(
            self.config,
            self.uploader,
            "User Upload",
            "document_id",
            "document_name",
            "project_id",
            "company_name",
            "ASSIGNMENT",
        )
        assert doc.status_code == "906"
        assert doc.is_classified() is False
        doc.update(
            self.config,
            self.uploader,
            {"document_date": "2000-12-31"},
        )
        assert doc.is_classified() is True
        assert doc.status_code.startswith("00")

    def test_current_status_for_multiple_identical(self) -> None:
        # it's possible to have loops in status transitions: 01a -> 01b -> 01a
        # test if current status returns the latest one for the given code
        doc = DocumentStatus(
            document_id="id",
            project_id="id",
            status_trail=[
                {"code": "001", "user": {"id": "01", "lan_id": "01", "role": ["rw"]}},
                {"code": "005", "user": {"id": "01", "lan_id": "01", "role": ["rw"]}},
                {"code": "001", "user": {"id": "pp", "lan_id": "pp", "role": ["xx"]}},
            ],
            status_code="001",
        )
        status = doc.current_status()
        assert status["user"]["lan_id"] == "pp"


class TestSchemaValidation:
    def test_valid(self) -> None:
        validate_json(
            {"company_name": "ACME", "source_type": "Email"},
            schemas.DOCUMENT_UPDATE_SCHEMA,
        )
        validate_json(
            {
                "project_id": "asdf",
                "version": 1,
                "status_map": {
                    "access_roles": [
                        "Workbench:CSSupportMaker",
                        "Workbench:CSSupportChecker",
                    ],
                    "statuses": [
                        {
                            "code": "001",
                            "header": "File created from original hard copy",
                            "description": "Original scanned and held by Central Services.",
                            "remarks": "optional",
                            "final": False,
                        }
                    ],
                    "transitions": {
                        "001": {
                            "001": {"Workbench:CSSupportMaker": ["BBCA", "non-BBCA"]},
                            "001Z": {
                                "Workbench:CSSupportMaker": ["BBCA"],
                                "Workbench:CSSupportChecker": ["non-BBCA"],
                            },
                        }
                    },
                },
            },
            schemas.CONFIG_SCHEMA,
        )

    def test_invalid(self) -> None:
        with pytest.raises(ValidationError):
            validate_json(
                {"document_id": "test1.pdf"},
                schemas.DOCUMENT_UPDATE_SCHEMA,
            )


class TestDocumentTypes:
    def setup_method(self) -> None:
        self.document_types = DocumentTypes(
            {
                "type1": {"bbca_documents": True},
                "type2": {"bbca_documents": False},
                "type3": {"bbca_documents": True},
            }
        )

    def test_all_types(self):
        assert self.document_types.all_types() == {"type1", "type2", "type3"}

    def test_is_bbca_type(self):
        assert self.document_types.is_bbca_type("type1") == True
        assert self.document_types.is_bbca_type("type2") == False
        assert self.document_types.is_bbca_type("type3") == True
