from datetime import datetime
from functools import wraps
from typing import TYPE_CHECKING

import jsonschema
import pytz
from flask import jsonify, request

if TYPE_CHECKING:
    from collections.abc import Callable
    from typing import Any

    from pytz.tzinfo import BaseTzInfo
    from werkzeug.wrappers import Response as WerkzeugResponse


def require_json_content_type(f: "Callable[..., Any]") -> "Callable[..., Any]":
    @wraps(f)
    def decorated_function(*args, **kwargs):  # type: ignore[no-untyped-def]
        # content-type might contain charset, so let's check mimetype instead:
        if request.mimetype != "application/json":
            return jsonify(message="Content-Type must be application/json"), 415
        return f(*args, **kwargs)

    return decorated_function


def response_json(
    status: str, msg: str, status_code: int
) -> "tuple[WerkzeugResponse, int]":
    res = jsonify({"status": status, "message": msg})
    res.status_code = status_code
    return res, status_code


def validate_json(obj: "dict[Any, Any]", schema: "dict[Any, Any]") -> None:
    """Raises jsonschema.exceptions.ValidationError."""
    jsonschema.validate(instance=obj, schema=schema)


def current_timestamp(tz: "BaseTzInfo" = pytz.utc) -> str:
    return datetime.now().astimezone(tz).strftime("%Y-%m-%d %H:%M:%S")


def utc_to_local(
    datetime_str: str,
    local_tz: "BaseTzInfo",
    display_format: str = "%Y-%m-%d %H:%M:%S",
) -> str:
    return (
        datetime.fromisoformat(datetime_str)
        .replace(tzinfo=pytz.utc)
        .astimezone(local_tz)
        .strftime(display_format)
    )
