# This module provides json schema definitions used for data validation.

from .document_types import DocumentTypes

doc_types = DocumentTypes.load()

DOC_SOURCE = {
    "Email": "004",
    "User Upload": "005",
    "End point Upload": "006",
    "WFI Migration": "007",
}

DOCUMENTS_GET_BY_IDS_SCHEMA = {
    "type": "array",
    "minItems": 1,
    "maxItems": 10000,
    "items": {"type": "string"},
}

DOCUMENT_CREATE_SCHEMA = {
    "type": "object",
    "required": ["document_id", "document_name", "source_type", "updater"],
    "properties": {
        "document_id": {"type": "string", "maxLength": 50},
        "document_name": {"type": "string"},
        "source_type": {"enum": list(DOC_SOURCE.keys())},
        "retry_create": {"type": "boolean", "default": True},
        "company_name": {"type": "string", "minLenght": 2, "maxLength": 1024},
        "document_type": {"enum": list(doc_types.all_types())},
        "document_date": {
            "type": "string",
            "format": "date",
        },
        "updater": {
            "type": "object",
            "required": ["lan_id", "role_ocbc", "name"],
            "properties": {
                "lan_id": {
                    "type": "array",
                    "minItems": 1,
                    "maxItems": 1,
                    "items": {"type": "string", "maxLength": 250},
                },
                "email": {
                    "type": "array",
                    "minItems": 1,
                    "maxItems": 1,
                    "items": {"type": "string", "maxLength": 300},
                },
                "role_ocbc": {
                    "type": "array",
                    "minItems": 1,
                    # "maxItems": 2, ?
                    "items": {"type": "string", "maxLength": 250},
                },
                "name": {
                    "type": "array",
                    "minItems": 1,
                    "maxItems": 1,
                    "items": {"type": "string", "maxLength": 300},
                },
            },
        },
    },
}

DOCUMENTS_BATCH_CREATE_SCHEMA = {
    "type": "array",
    "minItems": 1,
    "maxItems": 10000,
    "items": {
        "type": "object",
        "properties": {
            "document_id": {"type": "string", "maxLength": 50},
            "document_name": {"type": "string"},
            "company_name": {"type": "string", "minLenght": 2, "maxLength": 1024},
            "document_type": {"enum": list(doc_types.all_types())},
            "document_date": {
                "type": "string",
                "format": "date",
            },
        },
        "required": ["document_id", "document_name"],
    },
}

DOCUMENT_UPDATE_SCHEMA = {
    "type": "object",
    "properties": {
        "company_name": {"type": "string", "minLenght": 2, "maxLength": 1024},
        "document_type": {"enum": list(doc_types.all_types())},
        "document_date": {
            "type": "string",
            "format": "date",
        },
    },
    "anyOf": [
        {"required": ["company_name"]},
        {"required": ["document_type"]},
        {"required": ["document_date"]},
    ],
}

STATUS_UPDATE_SCHEMA = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "type": "object",
    "properties": {
        "status_code": {"type": "string", "minLenght": 2, "maxLength": 6},
        "remarks": {"type": "string", "minLenght": 2, "maxLength": 1000},
    },
    "required": ["status_code"],
}

BULK_STATUS_UPDATE_SCHEMA = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "type": "object",
    "properties": {
        "status_code": {"type": "string", "minLenght": 2, "maxLength": 6},
        "remarks": {"type": "string", "minLenght": 2, "maxLength": 1000},
        "document_ids": {
            "type": "array",
            "minItems": 1,
            "items": {
                "type": "string",
                "maxLength": 50,
            },
        },
    },
    "required": ["status_code", "document_ids"],
}

CONFIG_SCHEMA = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "type": "object",
    "properties": {
        "project_id": {"type": "string", "maxLength": 50},
        "version": {"type": "number", "minimum": 1},
        "status_map": {
            "type": "object",
            "properties": {
                "access_roles": {
                    "type": "array",
                    "minItems": 1,
                    "items": {
                        "type": "string",
                        "minLenght": 2,
                        "maxLength": 256,
                    },
                },
                "statuses": {
                    "type": "array",
                    "minItems": 1,
                    "items": {
                        "type": "object",
                        "properties": {
                            "code": {
                                "type": "string",
                                "minLenght": 2,
                                "maxLength": 6,
                            },
                            "header": {
                                "type": "string",
                                "minLenght": 2,
                                "maxLength": 512,
                            },
                            "remarks": {
                                "enum": ["optional", "mandatory", "forbidden"],
                            },
                            "description": {
                                "type": "string",
                                "minLenght": 2,
                                "maxLength": 1024,
                            },
                            "final": {"type": "boolean"},
                        },
                        "required": ["code", "header", "remarks", "final"],
                    },
                },
            },
            # cant do it with jsonschema, contents depend on codes defined in statuses
            "transitions": {"type": "object"},
            "required": ["access_roles", "statuses", "transitions"],
        },
    },
    "required": ["project_id", "version", "status_map"],
}
