from datetime import datetime

import pytz
from fpdf import FPDF

from .util import utc_to_local


class PrintLabelPDF(FPDF):
    """Generates a PDF for all documents uploaded today for a single
    company.
    """

    def __init__(
        self,
        company_name: str,
        docs: "list[dict[str, str]]",
        company_cif: str = "",
        segment: str = "",
    ) -> None:
        super().__init__()
        self.company = company_name
        self.cif = company_cif
        self.segment = segment
        self.docs = docs
        self._render_pdf()

    def save_to_file(self, filepath: str = "print-label.pdf") -> None:
        self.output(filepath)

    def get_binary(self) -> bytearray:
        return self.output()

    def _render_pdf(self) -> None:
        self.add_page()
        self.set_left_margin(15)
        self.set_right_margin(15)
        self.set_auto_page_break(auto=True, margin=25)

        self._set_intro()
        self.ln()

        self.set_font("Helvetica", "", 7)
        with self.table(
            col_widths=(55, 30, 30, 35, 30),
            line_height=1.5 * self.font_size,  # type: ignore[arg-type]
            padding=1,
            text_align="LEFT",
        ) as table:
            # Table header
            row = table.row()
            row.cell("File Name")
            row.cell("Document Type")
            row.cell("Document Upload <Date/Time>")
            row.cell("CSO's Name")
            row.cell("Status Update by BLT <Lodgement/Return>")

            for doc in self.docs:
                row = table.row()
                row.cell(doc["title"])
                row.cell(doc["doc_type"])
                row.cell(
                    utc_to_local(
                        doc["created_at"],
                        pytz.timezone("Asia/Singapore"),
                        display_format="%d/%m/%Y %H:%M:%S",
                    )
                )
                row.cell(doc["cso_name"])
                row.cell("")  # Empty for 'Status Update'

    def _set_intro(
        self,
    ) -> None:
        self.set_text_color(0, 0, 0)
        self.set_font("Helvetica", "BU", 16)
        self.cell(180, 16, "LIST OF DOCUMENTS", align="C")

        self.ln(20)

        self.set_font("Helvetica", "B", 10)
        self.cell(30, 6, "Borrower Name:", border=0)
        self.set_font("Helvetica", "BI", 10)
        self.cell(30, 6, self.company, border=0)
        self.ln()

        self.set_font("Helvetica", "B", 10)
        self.cell(30, 6, "CIF:", border=0)
        self.set_font("Helvetica", "BI", 10)
        self.cell(30, 6, self.cif, border=0)
        self.ln()

        self.set_font("Helvetica", "B", 10)
        self.cell(30, 6, "Segment:", border=0)
        self.set_font("Helvetica", "BI", 10)
        self.cell(30, 6, self.segment, border=0)
        self.ln()

    def header(self) -> None:
        self.set_font("Helvetica", "B", 8)
        self.cell(
            0,
            4,
            "CCA BBCA",
            align="L",
        )
        self.cell(
            0,
            4,
            "Generated on",
            align="R",
        )
        self.ln()
        self.cell(
            0,
            4,
            datetime.now()
            .astimezone(pytz.timezone("Asia/Singapore"))
            .strftime("%d/%m/%Y %H:%M:%S"),
            align="R",
        )
        self.ln(10)

    def footer(self) -> None:
        self.set_y(-15)  # Position at 1 cm from bottom:
        self.set_font("Helvetica", "", 7)
        self.cell(
            0,
            6,
            "Internal",
            align="C",
        )
