import logging
from typing import TYPE_CHECKING

import requests
from status_tracking.util import validate_json

from .schemas import DOCUMENT_CREATE_SCHEMA

if TYPE_CHECKING:
    from configparser import ConfigParser
    from typing import Any


STATUS_TRACKING_URL = (
    "{cluster}/studio/document_status_tracking/projects/"
    "{project_id}/documents?token={token}"
)
log = logging.getLogger(__name__)


class DocCreationErrorHandler:
    """Tries to create a status tracking record."""

    def __init__(self, octopus_config: "ConfigParser") -> None:
        self.octopus_config = octopus_config
        self.timeout = float(
            self.octopus_config.get(
                "status_tracking", "http_request_timeout_seconds", fallback="10"
            )
        )

    # each invocation happens in a newly created process so it's not possible
    # to keep connections open (db, http, ...)
    def retry_document_create(self, create_doc_command: "dict[str,Any]") -> None:
        validate_json(create_doc_command, DOCUMENT_CREATE_SCHEMA)
        document_id: str = create_doc_command.get(
            "document_id"  # validate_json makes sure this exists and is str
        )  # type: ignore[assignment]
        log.info("retry handler processing message %s", document_id)
        self.create_document(create_doc_command)
        log.info("document sucessfully synced %s", document_id)

    def create_document(self, create_doc_command: "dict[str,Any]") -> None:
        # prevent a loop: call create -> fails -> new item enqueued -> repeat...
        create_doc_command["retry_create"] = False
        res = requests.post(
            STATUS_TRACKING_URL.format(
                cluster=self.octopus_config["squirro"]["cluster"],
                project_id=self.octopus_config["squirro"]["project_id"],
                token=self.octopus_config["squirro"]["token"],
            ),
            json=create_doc_command,
            timeout=self.timeout,
        )
        res.raise_for_status()
        log.info("created document %s", create_doc_command["document_id"])
        log.debug(res.text)
