"""Email utility for status tracking."""

import logging
from datetime import datetime
from typing import TYPE_CHECKING

import pytz

from octopus.data import CompanyDataIndex
from octopus.email import STREAM_NAME, load_email_config, EmailPayload
from octopus.stream import add_to_stream

if TYPE_CHECKING:
    from typing import Any

    from redis import StrictRedis

    from .models import DocumentStatus


def construct_email_payload(
    doc: "DocumentStatus", config: "dict[str, Any]"
) -> "dict[str, Any]":
    return {
        "data": {
            "id": doc.document_id,
            "labels": {
                "company_name": doc.company_name,
                "document_date": doc.document_date,
                "document_id": doc.document_id,
                "document_name": doc.document_name,
                "document_type": doc.document_type,
            },
            "originator": doc.status_trail[0]["user"]["email"],
            "status": doc.status_trail[-1],
            "upload_timestamp": doc.status_trail[0]["timestamp"],
            "url": (
                f"{config['squirro']['url']}/app/#dashboard/"
                f"{config['squirro']['project_id']}?"
                f"modal-item-id={doc.document_id}"
            ),
        },
        "type": "status_update",
    }


def _process_status_update_payload(payload: "dict[str, Any]") -> None:
    logging.debug("Processing status update payload.")
    company_index = CompanyDataIndex.load_index()
    recipients = set()

    config = load_email_config()
    if not (targets := config.triggers.get(payload["data"]["status"]["code"])):
        logging.info("No email sending is required.")
        payload.clear()
        return

    if "originator" in targets:
        recipients.add(payload["data"]["originator"])
    if "blt_distribution" in targets:
        recipients.add(config.distribution_list["blt_distribution"])
    if "business_support" in targets:
        recipients.add(config.distribution_list["business_support"])
    if company_name := payload["data"]["labels"].get("company_name", ""):
        company = company_index.search_by_names(company_name)[0]
        payload["data"]["labels"]["rm_name"] = company.get("rm_name", [""])[0]
        payload["data"]["labels"]["team_name"] = company.get(
            "wfi_company_team_name", [""]
        )[0]
        payload["data"]["labels"]["segment"] = company.get(
            "company_segment_full", [""]
        )[0]
        if "rm" in targets:
            if rm_email := company.get("rm_email", [""])[0]:
                recipients.add(rm_email)
            if "rm_prev" in targets and (
                (
                    old_company := payload["data"]["status"]["description"]
                    .split("From ")[1]
                    .split(" to ")[0]
                )
                and (data := company_index.search_by_names(old_company))
                and (rm_email_old := data[0].get("rm_email", [""])[0])
            ):
                recipients.add(rm_email_old)
        if "bbca_distribution" in targets and (
            segment := company.get("wfi_company_segment")
        ):
            recipients.add(
                config.distribution_list[f"bbca_distribution_{segment[0].lower()}"]
            )
    payload["recipients"] = ", ".join(recipients)

    # Convert "2024-04-30 16:15:53" which is UTC to SGT timestamp
    payload["data"]["status"]["timestamp"] = (
        datetime.strptime(payload["data"]["status"]["timestamp"], "%Y-%m-%d %H:%M:%S")
        .astimezone(pytz.timezone("Asia/Singapore"))
        .strftime("%Y-%m-%d %H:%M:%S")
    )
    payload["data"]["upload_timestamp"] = (
        datetime.strptime(payload["data"]["upload_timestamp"], "%Y-%m-%d %H:%M:%S")
        .astimezone(pytz.timezone("Asia/Singapore"))
        .strftime("%Y-%m-%d %H:%M:%S")
    )


def send_email(
    payload: "dict[str, Any] | list[dict[str, Any]]", redis_client: "StrictRedis[bytes]"
) -> None:
    """Generates email payload for the given document status."""
    if not isinstance(payload, list):
        payload = [payload]

    for p in payload:
        _process_status_update_payload(p)
        if p:
            add_to_stream(EmailPayload.create_payload(**p).to_dict(), STREAM_NAME, redis_client=redis_client)
