# Status update permissions depends on document category: bbca vs non-bbca.
# This module loads document type definitions and provides category mapping.

import json
import logging
import sys

from octopus.clients.redis_client import init_redis_client

log = logging.getLogger(__name__)
redis = init_redis_client() if "pytest" not in sys.modules else None


class DocumentTypes:
    """All document types along with their category (BBCA)"""

    def __init__(self, mappings: "dict[str, dict[str, str|bool]]") -> None:
        self.mapping = mappings
        self.bbca_types = {
            k
            for k in mappings.keys()
            if "bbca_documents" in mappings[k] and mappings[k]["bbca_documents"]
        }

    @staticmethod
    def load() -> "DocumentTypes":
        if not redis:
            return DocumentTypes({})
        log.info("loading document mappings from redis")
        hash_bytes: dict[bytes, bytes] = redis.hgetall("document_type_mapping_hash")
        data = {k.decode(): json.loads(v) for k, v in hash_bytes.items()}
        return DocumentTypes(data)

    def all_types(self) -> "set[str]":
        return set(self.mapping.keys())

    def is_bbca_type(self, doctype: str) -> "bool":
        return doctype in self.bbca_types
