# pylint: disable=too-many-lines
import logging
import sys
import threading
from datetime import datetime
from io import BytesIO
from os import chdir, path
from typing import TYPE_CHECKING, TypedDict

from alembic import command as alembic_command
from alembic.config import Config as AlembicConfig
from flask import Response, jsonify, render_template, request, send_file
from flask import session as flask_session
from jsonschema.exceptions import ValidationError
from rq import Queue, Retry
from sqlalchemy import exc as sql_exc
from sqlalchemy import func
from status_tracking.email import construct_email_payload, send_email
from status_tracking.error_handlers import DocCreationErrorHandler
from status_tracking.errors import (
    AccessNotAllowed,
    DocumentNotFound,
    InvalidRequest,
    SquirroProjectNotFound,
    UserNotFound,
)
from status_tracking.label import PrintLabelPDF
from status_tracking.models import (
    DocumentStatus,
    DocumentUploader,
    StatusTrackingConfig,
    StatusTrailPDF,
    Updater,
    User,
)
from status_tracking.queue_error_handler import RedisQueueProcessor
from status_tracking.schemas import (
    BULK_STATUS_UPDATE_SCHEMA,
    CONFIG_SCHEMA,
    DOCUMENT_CREATE_SCHEMA,
    DOCUMENT_UPDATE_SCHEMA,
    DOCUMENTS_BATCH_CREATE_SCHEMA,
    DOCUMENTS_GET_BY_IDS_SCHEMA,
    STATUS_UPDATE_SCHEMA,
)
from status_tracking.util import (
    current_timestamp,
    require_json_content_type,
    response_json,
    validate_json,
)

from octopus.clients import init_redis_client, init_squirro_client, init_wfi_client
from octopus.utils import load_config
from squirro.common.dependency import get_injected, register_instance
from squirro.integration.frontend.context import execute_in_studioaware_context
from squirro.sdk.studio import StudioPlugin
from squirro_client import exceptions as squirro_client_exceptions

if TYPE_CHECKING:
    from typing import Any, NotRequired

plugin = StudioPlugin(__name__)
log = logging.getLogger(__name__)

DEFAULTS: "dict[str, str]" = {
    "error_redis_queue_name": "status_tracking_sync",
    "error_retry_max_attempts": "100",
    "error_retry_ttl_seconds": "600000",
    "error_retry_interval_seconds_csv": "60,120,300",
}

# couldn't find a list on the net, manually compiled one by going through
# https://docs.sqlalchemy.org/en/14/core/exceptions.html
TRANSIENT_EXCEPTIONS = (
    sql_exc.DisconnectionError,
    sql_exc.OperationalError,
    sql_exc.InternalError,
    sql_exc.TimeoutError,
    squirro_client_exceptions.ConnectionError,
    squirro_client_exceptions.RetryError,
    squirro_client_exceptions.TransportError,
)

# get_injected has no user context when triggered from pipelet
# since an instance of squirro_client is needed when documents are uploaded
# we keep it consistent by initializing it here to avoid multiple initializations
sqclient_no_user_context, _ = init_squirro_client()
wfi_client = init_wfi_client()


def run_db_migrations() -> None:
    ini_path = f"{path.dirname(__file__)}{path.sep}alembic.ini"
    log.info("init::running migrations - %s", ini_path)
    # alembic expects to be run from the same directory as alembic.ini:
    chdir(path.dirname(__file__))
    alembic_config = AlembicConfig(ini_path)
    alembic_command.upgrade(alembic_config, "head")
    log.info("init::migrations completed")


with plugin.app_context():
    run_db_migrations()

octopus_config = load_config()
RETRY_TTL_SECONDS = int(
    octopus_config.get("status_tracking", "error_retry_ttl_seconds", vars=DEFAULTS)
)
RETRY_MAX_ATTEMPTS = int(
    octopus_config.get("status_tracking", "error_retry_max_attempts", vars=DEFAULTS)
)
RETRY_INTERVALS_SECONDS = list(
    map(
        int,
        octopus_config.get(
            "status_tracking",
            "error_retry_interval_seconds_csv",
            vars=DEFAULTS,
        ).split(","),
    )
)
error_queue = octopus_config.get(
    "status_tracking", "error_redis_queue_name", vars=DEFAULTS
)
redis_client = init_redis_client()
consumer = RedisQueueProcessor(
    redis=redis_client,
    queueus=[error_queue],
)
register_instance("consumer", consumer)
# run_background_processes(
#     count=1,
#     queues=None,
#     pool=None,
#     task_processor_cls=None,
#     name=error_queue,
# )
errors_queue = Queue(error_queue, connection=redis_client)
err_handler = DocCreationErrorHandler(octopus_config)

# pytest only works with skip_authentication=True
# use skip_authentication=True if using pytest
plugin_options: "dict[str, bool]" = (
    {"skip_authentication": True}
    if "pytest" in sys.modules
    else {"allow_project_readers": True}
)


@plugin.route("/projects/<project_id>/config", methods=["GET"], **plugin_options)  # type: ignore[misc]
def config_get(project_id: str) -> "tuple[Response, int]":
    latest = _load_config(project_id)
    response = jsonify({"config": latest.to_dict()})
    return (response, 200)


def _load_config(project_id: str) -> "StatusTrackingConfig":
    log.info("load config %s", project_id)
    session = get_injected("db_studio")
    latest = (
        session.query(StatusTrackingConfig)
        .filter_by(project_id=project_id)
        .order_by(StatusTrackingConfig.version.desc())
        .first()
    )
    log.debug(latest)
    if not latest:
        raise SquirroProjectNotFound(f"{project_id} not found")
    return latest


@plugin.route(
    "/projects/<project_id>/config",
    methods=["POST"],  # only allow admins
)  # type: ignore[misc]
@require_json_content_type  # type: ignore[misc]
def config_update(project_id: str) -> "tuple[Response, int]":
    # require_json_content_type decorator enforces correct mimetype:
    request_data: dict[str, Any] = request.get_json()  # type: ignore[assignment]
    validate_json(request_data, CONFIG_SCHEMA)
    status_flow = StatusTrackingConfig(
        project_id=project_id,
        version=request_data.get("version"),
        status_map=request_data.get("status_map"),
    )
    session = get_injected("db_studio")
    try:
        session.add(status_flow)
        session.commit()
        # status_flow will now have id populated by sqlalchemy
        return (
            jsonify(
                {
                    "status": "Status tracking process definition updated successfully",
                    "config": status_flow.to_dict(),
                }
            ),
            201,
        )
    except sql_exc.IntegrityError as e:
        log.exception(e)
        invalid_req: tuple[Response, int] = response_json(
            "failure",
            str(e),
            400,
        )

        return invalid_req
    except Exception as e:  # pylint: disable-msg=W0718
        log.exception(e)
        server_err: tuple[Response, int] = response_json(
            "failure",
            str(e),
            500,
        )
        return server_err


@plugin.route("/admin/documents", methods=["POST"])  # type: ignore[misc]
def admin_document_create() -> "tuple[Response, int]":
    """Admin endpoint to create documents using a background queue.

    Document must be ingested into ElasticSearch for this to work.
    """
    request_data: dict[str, Any] = request.get_json()  # type: ignore[assignment]
    log.debug(request_data)
    validate_json(request_data, DOCUMENT_CREATE_SCHEMA)
    job = errors_queue.enqueue(
        err_handler.retry_document_create,
        request_data,
        ttl=RETRY_TTL_SECONDS,
        retry=Retry(
            max=RETRY_MAX_ATTEMPTS,
            interval=RETRY_INTERVALS_SECONDS,
        ),
    )
    return jsonify({"message": "Job accepted", "job_id": job.get_id()}), 200


@plugin.route("/projects/<project_id>/documents", **plugin_options)  # type: ignore[misc]
def documents_get(
    project_id: str,  # pylint: disable=unused-argument
) -> "tuple[Response, int]":
    args = request.args
    session = get_injected("db_studio")
    if session.bind.dialect.name == "mysql":
        timestamp = func.json_extract(DocumentStatus.status_trail, "$[0].timestamp")
    else:  # PostgreSQL
        timestamp = func.json_extract_path_text(
            DocumentStatus.status_trail,
            "0",
            "timestamp",
        )

    filter_criteria = []
    if statuses := args.getlist("statuses"):
        filter_criteria.append(DocumentStatus.status_code.in_(statuses))
    if created_before := args.get("created_before"):
        filter_criteria.append(timestamp < created_before)
    if created_after := args.get("created_after"):
        filter_criteria.append(timestamp > created_after)

    docs = session.query(DocumentStatus).filter(*filter_criteria).all()

    log.debug("Found %i docs", len(docs))

    return jsonify([doc.to_dict() for doc in docs]), 200


@plugin.route(
    "/projects/<project_id>/documents/ids", methods=["POST"], **plugin_options
)  # type: ignore[misc]
def documents_get_by_ids(
    project_id: str,  # pylint: disable=unused-argument
) -> "tuple[Response, int]":
    """Online reports allow users to filter by segment and rm name, and these
    fields are only available in ES, not in DB.

    The only way to get the documents that match the criteria is by
    filtering using the IDs returned from ES query.
    """
    ids: list[str] = request.get_json()  # type: ignore[assignment]
    validate_json(ids, DOCUMENTS_GET_BY_IDS_SCHEMA)

    session = get_injected("db_studio")
    docs = session.query(DocumentStatus).filter(DocumentStatus.document_id.in_(ids))
    return jsonify([doc.to_dict() for doc in docs]), 200


@plugin.route("/projects/<project_id>/documents", methods=["POST"], **plugin_options)  # type: ignore[misc]
@require_json_content_type  # type: ignore[misc]
def document_create(project_id: str) -> "tuple[Response, int]":
    # require_json_content_type decorator enforces correct mimetype:
    request_data: dict[str, Any] = request.get_json()  # type: ignore[assignment]
    log.debug(request_data)
    validate_json(request_data, DOCUMENT_CREATE_SCHEMA)
    config = _load_config(project_id)

    user: DocumentUploader = request_data.get("updater")
    uploader: Updater = {
        "lan_id": user["lan_id"][0],
        "email": user["email"][0],
        "name": user["name"][0],
        "role": user["role_ocbc"],
    }
    config.raise_not_allowed_to_access_status_tracking({"role_ocbc": uploader["role"]})
    doc_record = DocumentStatus.create(
        config,
        uploader,
        request_data.get("source_type"),
        document_id=request_data.get("document_id"),
        document_name=request_data.get("document_name"),
        project_id=project_id,
        company_name=request_data.get("company_name"),
        document_type=request_data.get("document_type"),
        document_date=request_data.get("document_date"),
        wfi_document_id=request_data.get("wfi_document_id"),
    )

    session = get_injected("db_studio")
    session.add(doc_record)
    session.flush()
    try:
        updated_keywords = {
            "current_doc_status": [doc_record.status_code],
            "current_doc_status_header": [
                config.codes[doc_record.status_code]["header"]
            ],
        }

        if doc_record.status_code != "007":  # Ignore docs migrated from WFI
            updated_keywords.update(
                {
                    "prev_doc_status": ["000"],
                    "prev_updater": [uploader["email"]],
                    "uploader_role": uploader["role"],
                }
            )
        sqclient_no_user_context.modify_item(
            project_id=project_id,
            item_id=doc_record.document_id,
            keywords=updated_keywords,
        )
        session.commit()
    except TRANSIENT_EXCEPTIONS as e:
        log.exception(e)
        session.rollback()
        if not request_data.get("retry_create", False):
            raise e
        job = errors_queue.enqueue(
            err_handler.retry_document_create,
            request_data,
            ttl=RETRY_TTL_SECONDS,
            retry=Retry(
                max=RETRY_MAX_ATTEMPTS,
                interval=RETRY_INTERVALS_SECONDS,
            ),
        )
        log.warning(
            {
                "msg": "Failed to create document, created a background job to recover",
                "job_id": job.get_id(),
            }
        )
        return (
            jsonify({"message": "Document not yet created, request accepted"}),
            202,
        )

    # Send email once status is updated
    threading.Thread(
        target=send_email,
        args=(construct_email_payload(doc_record, octopus_config), redis_client),
    ).start()

    # no need for next_statuses because this call comes from the pipelet
    return (
        jsonify(
            {
                "message": "Document created",
                "document": doc_record.to_dict(),
            }
        ),
        201,
    )


class DocumentBatchCreate(TypedDict):
    """Represents request to create status tracking record for previously
    ingested documents.
    """

    document_id: str
    source_type: str
    company_name: "NotRequired[str]"
    document_type: "NotRequired[str]"
    document_date: "NotRequired[str]"


@plugin.route(
    "/projects/<project_id>/documents-batch", methods=["POST"], **plugin_options
)  # type: ignore[misc]
@require_json_content_type  # type: ignore[misc]
def documents_batch_create(project_id: str) -> "tuple[Response, int]":
    """This method first commits the entire batch to DB and then updates
    elastic in batches (of 1000 items at a time).

    Case something goes wrong, you should manually remove documents from SQL
    and then retry:
        DELETE FROM document_status
            WHERE status_code = '008' AND DATE(updated) = CURDATE();
    """
    session = get_injected("db_studio")
    request_data: list[DocumentBatchCreate] = (
        # require_json_content_type decorator enforces correct type:
        request.get_json()  # type: ignore[assignment]
    )
    validate_json(request_data, DOCUMENTS_BATCH_CREATE_SCHEMA)

    session = get_injected("db_studio")
    status = "008"
    ts_now = current_timestamp()
    config = _load_config(project_id)
    batch = [
        DocumentStatus(
            document_id=d["document_id"],
            project_id=project_id,
            status_trail=[
                {
                    "code": status,
                    "user": {
                        "id": "System",
                        "name": "System",
                        "lan_id": "System",
                        "role": ["System"],
                    },
                    "header": config.codes[status]["header"],
                    "timestamp": ts_now,
                }
            ],
            company_name=d["company_name"],
            document_type=d["document_type"],
            document_date=d["document_date"],
            status_code=status,
        )
        for d in request_data
    ]
    for chunk in range(0, len(batch), 1000):
        session.add_all(batch[chunk : chunk + 1000])
        session.flush()
    session.commit()
    for chunk in range(0, len(batch), 1000):
        items = [
            {
                "id": d.document_id,
                "keywords": {
                    "current_doc_status": [d.status_code],
                    "current_doc_status_header": [config.codes[status]["header"]],
                },
            }
            for d in batch[chunk : chunk + 1000]
        ]
        sqclient_no_user_context.modify_items(project_id, items)
    return (
        jsonify({"message": "Documents updated"}),
        201,
    )


def _load_doc(document_id: str) -> "DocumentStatus":
    log.debug("load document %s", document_id)
    session = get_injected("db_studio")
    doc = session.query(DocumentStatus).get(document_id)
    if not doc:
        raise DocumentNotFound(document_id)
    log.debug(doc)
    return doc


@plugin.route(
    "/projects/<project_id>/documents/<document_id>", methods=["GET"], **plugin_options
)  # type: ignore[misc]
def document_get(project_id: str, document_id: str) -> "tuple[Response, int]":
    doc = _load_doc(document_id)
    caller = _get_caller()
    config = _load_config(project_id)
    config.raise_not_allowed_to_access_status_tracking(caller["user_information"])
    #  provide controls for possible updates:
    doc.next_statuses = config.list_available_followup_statuses(
        caller["user_information"], doc
    )
    return (jsonify({"document": doc.to_dict()}), 200)


@plugin.route(
    "/projects/<project_id>/documents/<document_id>/label/pdf",
    methods=["GET"],
    **plugin_options,
)  # type: ignore[misc]
def document_download_print_label(
    project_id: str,
    document_id: str,  # pylint: disable=W0613
) -> "Any":
    log.debug("download print label for %s, %s", document_id, project_id)
    doc = _load_doc(document_id)
    if not doc.company_name:
        raise InvalidRequest("Company name not set")

    # get all documents for this company
    sqclient = get_injected("squirro_client")
    midnight = datetime.now().replace(hour=0, minute=0, second=0).isoformat()
    items = sqclient.query(
        project_id,
        query=f'-is_deleted:true AND company_name:"{doc.company_name}"',
        created_after=midnight,
        fields=["title", "created_at", "keywords"],
        # assume they wont upload more in a day
        # otherwise have to do paging
        count=10000,
    )

    if not items["total"]:
        return (
            render_template(
                "no-results.html",
                error_msg="Sorry, we couldn't find any results for your document. "
                "Please try again with a different document.",
            ),
            400,
        )

    def key_to_str(key: str, item: "dict[str,Any]") -> str:
        return ",".join(item["keywords"][key]) if key in item["keywords"] else ""

    # need the header first:
    sqdoc = [item for item in items["items"] if item["id"] == document_id][0]
    segment = key_to_str("wfi_company_segment", sqdoc)
    cif = key_to_str("company_cif", sqdoc)

    uploaded_today = []
    for item in items["items"]:
        doc_type = key_to_str("document_type", item)
        cso_name = key_to_str("cso_name", item)
        uploaded_today.append(
            {
                "title": item["title"],
                "created_at": item["created_at"],
                "doc_type": doc_type,
                "cso_name": cso_name,
            }
        )

    pdf = PrintLabelPDF(doc.company_name, uploaded_today, cif, segment)
    return send_file(
        BytesIO(pdf.get_binary()),
        as_attachment=False,
        mimetype="application/pdf",
    )


@plugin.route(
    "/projects/<project_id>/documents/<document_id>/status/pdf",
    methods=["GET"],
    **plugin_options,
)  # type: ignore[misc]
def document_get_status(
    project_id: str,
    document_id: str,  # pylint: disable=W0613
) -> "Any":
    log.debug("status trail pdf for %s", document_id)
    caller = _get_caller()
    config = _load_config(project_id)
    config.raise_not_allowed_to_access_status_tracking(caller["user_information"])

    doc = _load_doc(document_id)

    sort_order = request.args.get("sort", default="DESC")
    sqclient = get_injected("squirro_client")
    sqitem = sqclient.get_item(project_id, document_id, fields="title,keywords")
    filename = sqitem["item"]["title"]
    keywords = sqitem["item"]["keywords"]
    cif = ", ".join(keywords.get("company_cif", []))
    doc_format = keywords.get("document_format", [""])[0]
    pdf = StatusTrailPDF(doc, filename, cif, doc_format, sort_order)
    if filename:
        filename_without_ext, _ = path.splitext(filename)
        pdf_filename = f"{filename_without_ext}-{sort_order}-status-history.pdf"
    else:
        pdf_filename = f"{document_id}-{sort_order}-status-history.pdf"

    return send_file(
        BytesIO(pdf.get_binary()),
        as_attachment=True,
        download_name=pdf_filename,
        mimetype="application/pdf",
    )


@plugin.route(
    "/projects/<project_id>/documents/<document_id>/status",
    methods=["POST"],
    **plugin_options,
)  # type: ignore[misc]
@require_json_content_type  # type: ignore[misc]
# pylint: disable=too-many-locals
def document_update_status(project_id: str, document_id: str) -> "tuple[Response, int]":
    # require_json_content_type decorator enforces correct mimetype:
    status_update_req: dict[str, Any] = request.get_json()  # type: ignore[assignment]
    validate_json(status_update_req, STATUS_UPDATE_SCHEMA)

    doc = _load_doc(document_id)
    caller = _get_caller()
    config = _load_config(project_id)
    status = config.codes[status_update_req["status_code"]]
    if status["remarks"] == "mandatory" and "remarks" not in status_update_req:
        err_remark: tuple[Response, int] = response_json(
            "failure",
            f"Remarks is mandatory for {status['header']}",
            400,
        )
        return err_remark
    user_info = caller["user_information"]
    if not config.is_status_update_allowed(
        doc,
        status_update_req["status_code"],
        user_info,
    ):
        err_auth: tuple[Response, int] = response_json(
            "failure",
            f"You are not authorized to update {doc.status_code} "
            + f"to {status_update_req['status_code']}",
            403,
        )
        return err_auth

    role = user_info["role_ocbc"]
    if doc.status_code == "001Z" and status["code"] == "003B":
        for trail in doc.status_trail:
            if trail["user"]["email"] == user_info["email"][0]:
                # Updated by maker of that particular case
                if trail["user"]["role"][0] == "Workbench:CSSupportMaker":
                    role = ["Workbench:CSSupportMaker"]
                # Updated by checker of that particular case
                elif trail["user"]["role"][0] == "Workbench:CSSupportChecker":
                    role = ["Workbench:CSSupportChecker"]
                break
        else:
            # Updated by a third party (CS staff) who is not a maker/checker
            role = ["Workbench:CSSupportMaker&Checker"]
    elif len(user_info["role_ocbc"]) > 1:
        # If user is a dual role user, we only show the roles that are relevant to the
        # current status transition on the status trail. This is to avoid confusion for
        # the user.
        role = [
            x
            for x in config.status_map["transitions"][doc.status_code][
                status["code"]
            ].keys()
            if x in user_info["role_ocbc"]
        ]

    # Design decision: store all user attributes, from the SSO object,
    # (e.g. make a copy).
    # upside: we don’t need to query users when generating reports, or show statuses.
    #         It will also nicely work when users get decommissioned.
    # downside: it’s a duplicate, if user’s name change, here it will remain unchanged.
    doc.status_trail = doc.status_trail + [
        {
            "code": status["code"],
            "header": status["header"],
            "description": status["description"],
            "timestamp": current_timestamp(),
            "remarks": (
                status_update_req["remarks"]
                if "remarks" in status_update_req and status["remarks"] != "forbidden"
                else None
            ),
            "user": {
                "lan_id": user_info["lan_id"][0] if "lan_id" in user_info else None,
                "email": user_info["email"][0] if "email" in user_info else None,
                "name": caller["full_name"],
                "role": role,
            },
        }
    ]
    if not status["code"].startswith("AT"):
        doc.status_trail_report_idx = len(doc.status_trail) - 1
    prev_status_code = doc.status_code
    doc.status_code = status["code"]
    doc.updated = datetime.utcnow()
    sqclient = get_injected("squirro_client")
    session = get_injected("db_studio")
    # best-effort attempt to keep sql and elastic in sync:
    session.flush()  # this makes sure db connection works
    try:
        updated_keywords = {
            "current_doc_status": [doc.status_code],
            "current_doc_status_header": [status["header"]],
            "prev_doc_status": [
                "000" if prev_status_code.startswith("9") else prev_status_code
            ],
            "prev_updater": user_info.get("email", []),
        }
        if doc.status_code == "001D":
            updated_keywords["is_deleted"] = ["true"]
            # Update in WFI - Soft Delete [ODST-34] - StatusMap Component
            # RUN before sqclient.modify_item to
            # catch WFI exceptions before update squirro item
            wfi_client.soft_delete(doc.wfi_document_id)
        sqclient.modify_item(
            project_id=project_id,
            item_id=document_id,
            keywords=updated_keywords,
        )
        # at this point we have elastic updated but sql not yet persisted to disk:
        session.commit()
        # tx commit can fail at which point we have an inconsistency between
        # elastic and sql, but since sql is the source of truth it's not that critical.
        # Users will get an error message and can retry status update.
    except Exception as e:
        log.exception(e)
        session.rollback()
        raise e

    # Send email once status is updated
    threading.Thread(
        target=send_email,
        args=(construct_email_payload(doc, octopus_config), redis_client),
    ).start()

    # provide controls for possible updates:
    doc.next_statuses = config.list_available_followup_statuses(user_info, doc)
    return (jsonify({"message": "Status updated", "document": doc.to_dict()}), 201)


def _bulk_update_status(  # pylint: disable=R0914
    project_id: str, status_update_req: "dict[str,Any]", config: StatusTrackingConfig
) -> "tuple[Response, int]":
    status = config.codes[status_update_req["status_code"]]
    caller = _get_caller()
    user_info = caller["user_information"]

    new_status_code_audit = [
        {
            "code": status["code"],
            "header": status["header"],
            "description": status["description"],
            "timestamp": current_timestamp(),
            "remarks": (
                status_update_req["remarks"]
                if "remarks" in status_update_req and status["remarks"] != "forbidden"
                else None
            ),
            "user": {
                "lan_id": user_info["lan_id"][0] if "lan_id" in user_info else None,
                "email": user_info["email"][0] if "email" in user_info else None,
                "name": caller["full_name"],
                "role": user_info["role_ocbc"],
            },
        }
    ]
    squirro_items = []
    updated_at = datetime.utcnow()
    session = get_injected("db_studio")
    docs = (
        session.query(DocumentStatus)
        .filter(DocumentStatus.document_id.in_(status_update_req["document_ids"]))
        .all()
    )

    status_before_update = docs[0].status_code

    payloads = []
    for doc in docs:
        # Check if all the documents have the same status code. This is required because
        # after a user bulk assigns the status, changes won't be reflected in another
        # user's browser until refresh. Possibility of documents with different statuses
        if doc.status_code != status_before_update:
            session.rollback()
            log.error(
                "Document %s has a different status %s. Expected %s.",
                doc.document_id,
                doc.status_code,
                status_before_update,
            )
            res: tuple[Response, int] = response_json(
                "failure",
                "Document statuses are outdated. "
                "Please refresh to fetch the latest statuses.",
                400,
            )
            return res

        doc.status_trail = doc.status_trail + new_status_code_audit
        doc.status_code = status["code"]
        doc.updated = updated_at

        # Update status trail report index
        if not doc.status_trail[-1]["code"].startswith("AT"):
            doc.status_trail_report_idx = len(doc.status_trail) - 1

        session.add(doc)
        updated_keywords = {
            "current_doc_status": [doc.status_code],
            "current_doc_status_header": [status["header"]],
            "prev_doc_status": [
                "000" if status_before_update.startswith("9") else status_before_update
            ],
            "prev_updater": user_info.get("email", []),
        }
        if doc.status_code == "001D":
            updated_keywords["is_deleted"] = ["true"]
            # Update in WFI - Soft Delete [ODST-34] - BulkAssign Component
            # RUN before sqclient.modify_items to
            # catch WFI exceptions before update squirro item
            wfi_client.soft_delete(doc.wfi_document_id)
        squirro_items.append(
            {
                "id": doc.document_id,
                "keywords": updated_keywords,
            }
        )
        payloads.append(construct_email_payload(doc, octopus_config))
    sqclient = get_injected("squirro_client")
    # best-effort attempt to keep sql and elastic in sync:
    session.flush()  # this makes sure db connection works
    try:
        sqclient.modify_items(
            project_id=project_id,
            items=squirro_items,
        )
        # at this point we have elastic updated but sql not yet persisted to disk:
        session.commit()
        # if commit fails we have an inconsistency between elastic and sql,
        # which in this case would have to be manually resolved.
    except Exception as e:
        log.exception(e)
        session.rollback()
        raise e

    threading.Thread(target=send_email, args=(payloads, redis_client)).start()

    return (jsonify({"message": "Status updated"}), 201)


@plugin.route(
    "/projects/<project_id>/documents-bulk-status", methods=["POST"], **plugin_options
)  # type: ignore[misc]
@require_json_content_type  # type: ignore[misc]
def documents_bulk_update_status(project_id: str) -> "tuple[Response, int]":
    """Update status for multiple documents at once.

    All documents should have the same status currently.
    """
    log.info("Bulk update status %s", request.get_json())
    # require_json_content_type decorator enforces correct mimetype:
    status_update_req: dict[str, Any] = request.get_json()  # type: ignore[assignment]
    validate_json(status_update_req, BULK_STATUS_UPDATE_SCHEMA)

    config = _load_config(project_id)
    status = config.codes[status_update_req["status_code"]]
    if status["remarks"] == "mandatory" and "remarks" not in status_update_req:
        err_remark: tuple[Response, int] = response_json(
            "failure",
            f"Remarks is mandatory for {status['header']}",
            400,
        )
        return err_remark

    caller = _get_caller()
    user_info = caller["user_information"]
    # since all have the same status it's enough to check permissions for one document:
    document_id = status_update_req["document_ids"][0]
    doc = _load_doc(document_id)
    if not config.is_status_update_allowed(
        doc,
        status_update_req["status_code"],
        user_info,
    ):
        err_auth: tuple[Response, int] = response_json(
            "failure",
            f"You are not authorized to update {doc.status_code} "
            + f"to {status_update_req['status_code']}",
            403,
        )
        return err_auth

    return _bulk_update_status(project_id, status_update_req, config)


@plugin.route(
    "/projects/<project_id>/documents/<document_id>", methods=["PUT"], **plugin_options
)  # type: ignore[misc]
@require_json_content_type  # type: ignore[misc]
def document_update(project_id: str, document_id: str) -> "tuple[Response, int]":
    """Update document's company_name, document_type or document_date."""
    # require_json_content_type decorator enforces correct mimetype:
    request_data: dict[str, Any] = request.get_json()  # type: ignore[assignment]
    validate_json(request_data, DOCUMENT_UPDATE_SCHEMA)
    caller = _get_caller()
    session = get_injected("db_studio")
    doc = _load_doc(document_id)

    user_info = caller["user_information"]
    updater = {
        "lan_id": user_info["lan_id"][0] if "lan_id" in user_info else None,
        "email": user_info["email"][0] if "email" in user_info else None,
        "name": caller["full_name"],
        "role": user_info["role_ocbc"],
    }
    config = _load_config(project_id)
    config.raise_not_allowed_to_access_status_tracking(caller["user_information"])
    status_before_update = doc.status_code
    doc.update(
        config,
        updater,
        attributes={
            "company_name": request_data.get("company_name"),
            "document_type": request_data.get("document_type"),
            "document_date": request_data.get("document_date"),
        },
    )
    if doc.status_code == status_before_update:
        session.commit()
    else:
        # document is now classified
        session.flush()
        try:
            updated_keywords = {
                "current_doc_status": [doc.status_code],
                "current_doc_status_header": [config.codes[doc.status_code]["header"]],
                "prev_doc_status": [
                    (
                        "000"
                        if status_before_update.startswith("9")
                        else status_before_update
                    )
                ],
                "prev_updater": user_info.get("email", []),
            }
            sqclient_no_user_context.modify_item(
                project_id=project_id,
                item_id=document_id,
                keywords=updated_keywords,
            )
            session.commit()
        except Exception as e:
            session.rollback()
            log.exception(e)
            raise e

    # Send email once status is updated
    threading.Thread(
        target=send_email,
        args=(construct_email_payload(doc, octopus_config), redis_client),
    ).start()

    # provide controls for possible updates:
    doc.next_statuses = config.list_available_followup_statuses(user_info, doc)
    return (
        jsonify(
            {
                "message": "Document updated",
                "document": doc.to_dict(),
            }
        ),
        200,
    )


###
# Utility methods
###


@execute_in_studioaware_context  # type: ignore[misc]
def _get_user_id() -> "Any| None":
    """Get the signed user from the frontend context (this plugin is its own
    flask app).
    """
    return flask_session.get("user_id")


def _get_caller() -> "User":
    """{
          id: 'redacted_id',
          tenant: 'project-name',
          email: 'john.doe@example.com',
          role: 'admin',
          role_permissions: [
            'admin',
            'admin_space',
            'profile.write.update',
            'projects.write.create'
          ],
          full_name: 'John Doe',
          authentications: {
            sso: {
              service_user: 'john.doe@example.com',
              display_name: null,
              access_token: null,
              access_token_expires: null,
              access_secret: null,
              access_secret_expires: null,
              state: 'ok'
            }
          },
          config: { timezone: 'UTC', json_preferences: '{}', user_store: '{}' },
          groups: [ { id: 'redacted_id', name: 'Admins' } ],
          user_information: {
            Role: [ 'Owner', 'Admins' ],
            email: [ 'john.doe@example.com' ],
            givenName: [ 'Jonh' ],
            surname: [ 'Doe' ],
            uid: [ 'john.doe@example.com' ]
    # OCBC specific attributes:
            "ecm_role": ["admin"],
            "job_title": ["WCM(WCM 80) RM"],
            "lan_id": ["A001"],
            "org_unit_id": ["RE078"],
            "org_unit_type": ["Team"],
            "rmcode": ["myecm1"],
            "role": ["RM"],
            "role_ocbc": ["Workbench:CSSupportMaker", "Workbench:CSSupportChecker"]
          }
        }
    """
    caller_id = _get_user_id()
    if not caller_id:
        raise UserNotFound("Failed to load user session")
    sqclient = get_injected("squirro_client")
    caller = sqclient.get_user_data(caller_id)
    if not caller:
        raise UserNotFound("Failed to load user session")
    return caller


###
# Error handlers
###


@plugin.errorhandler(ValidationError)  # type: ignore[misc]
def handle_json_validation_error(e: ValidationError) -> "tuple[Response, int]":
    log.exception(e)
    resp: tuple[Response, int] = response_json(
        "failure",
        f"Invalid request: {e}",
        400,
    )
    return resp


@plugin.errorhandler(SquirroProjectNotFound)  # type: ignore[misc]
def handle_squirro_project_not_found(
    e: SquirroProjectNotFound,
) -> "tuple[Response, int]":
    log.exception(e)
    resp: tuple[Response, int] = response_json(
        "failure",
        f"Project not found {e}",
        404,
    )
    return resp


@plugin.errorhandler(DocumentNotFound)  # type: ignore[misc]
def handle_squirro_document_not_found(e: DocumentNotFound) -> "tuple[Response, int]":
    log.exception(e)
    resp: tuple[Response, int] = response_json(
        "failure",
        f"Document not found {e}",
        404,
    )
    return resp


@plugin.errorhandler(UserNotFound)  # type: ignore[misc]
def handle_squirro_user_not_found(e: UserNotFound) -> "tuple[Response, int]":
    log.exception(e)
    resp: tuple[Response, int] = response_json(
        "failure",
        # perhaps do not leak any details for security reasons
        "Failed to load user session",
        500,
    )
    return resp


@plugin.errorhandler(InvalidRequest)  # type: ignore[misc]
def handle_invalid_request(e: InvalidRequest) -> "tuple[Response, int]":
    log.exception(e)
    resp: tuple[Response, int] = response_json(
        "failure",
        f"Invalid request: {e}",
        400,
    )
    return resp


@plugin.errorhandler(AccessNotAllowed)  # type: ignore[misc]
def handle_access_not_allowed(e: AccessNotAllowed) -> "tuple[Response, int]":
    log.exception(e)
    resp: tuple[Response, int] = response_json(
        "failure",
        f"Invalid request: {e}",
        403,
    )
    return resp
