"""Document type statistics plugin."""

import json
import logging
import sys
from typing import TYPE_CHECKING

from flask import jsonify

from octopus.clients import init_redis_client
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from typing import Any

    from flask import Response


log = logging.getLogger(__name__)
plugin = StudioPlugin(__name__)

# pytest only works with skip_authentication=True
# use skip_authentication=True if using pytest
# otherwise set allow_project_readers=True
plugin_options: "dict[str, Any]" = (
    {"skip_authentication": True}
    if "pytest" in sys.modules
    else {"allow_project_readers": True}
)

redis_client = init_redis_client()


@plugin.route("/doctype/<project_id>", **plugin_options)  # type: ignore[misc]
def doctype_stats(project_id: str) -> "tuple[Response, int]":
    """Get the document type statistics for a project.

    Returns:
        The document type statistics.
    """
    stats = redis_client.get(f"ml_accuracy_{project_id}")

    if not stats:
        log.error("ML accuracy stats not available.")
        return jsonify(success=False, results={}), 404

    stats = json.loads(stats.decode())
    return jsonify(success=True, results=stats), 200
