"""Autocomplete plugin."""

import logging
import sys
from typing import TYPE_CHECKING

from autocomplete.load import (
    load_company_names_autocomplete,
    load_cso_names_autocomplete,
    load_rm_names_autocomplete,
    regenerate_company_names_autocomplete,
    regenerate_cso_names_autocomplete,
    regenerate_rm_names_autocomplete,
)
from flask import jsonify, request

from octopus.clients import init_redis_client
from squirro.sdk.studio import StudioPlugin

if TYPE_CHECKING:
    from typing import Any

    from flask import Response


plugin = StudioPlugin(__name__)

# pytest only works with skip_authentication=True
# use skip_authentication=True if using pytest
# otherwise set allow_project_readers=True
plugin_options: "dict[str, Any]" = (
    {"skip_authentication": True}
    if "pytest" in sys.modules
    else {"allow_project_readers": True}
)

autocomplete_company_names = load_company_names_autocomplete()
autocomplete_cso_names = load_cso_names_autocomplete()
autocomplete_rm_names = load_rm_names_autocomplete()

redis_client = init_redis_client()


@plugin.route("/company_names/<project_id>", **plugin_options)  # type: ignore[misc]
def company_names(project_id: str) -> "Response":
    """Autocomplete company names.

    Returns:
        The response.
    """
    query: str = "" if not (q := request.args.get("q")) else q
    logging.info("Autocompleting company names for query `%s`", query)
    results = autocomplete_company_names(query)
    data: list[dict[str, str]] = []
    for result in list(results)[:30]:
        name: str
        cif: str
        name, cif = result.split("|")
        data.append({"name": name, "cif": cif})
    logging.info("Found %d results", len(results))
    return jsonify(data)


@plugin.route("/document_types/<project_id>", **plugin_options)  # type: ignore[misc]
def document_types(project_id: str) -> "Response":
    """Autocomplete document types.

    Returns:
        The response.
    """
    document_types: list[str] = []
    try:
        document_type_mapping: list[bytes] = redis_client.hkeys(
            "document_type_mapping_hash"
        )
    except Exception:
        logging.exception("Unable to retrieve document type mapping")
        raise

    if document_type_mapping:
        document_types = sorted([key.decode("utf-8") for key in document_type_mapping])

    return jsonify(document_types)


@plugin.route("/cso_names/<project_id>", **plugin_options)  # type: ignore[misc]
def cso_names(project_id: str) -> "Response":
    """Autocomplete CSO names.

    Returns:
        The response.
    """
    query: str = "" if not (q := request.args.get("q")) else q
    logging.info("Autocompleting cso names for query `%s`", query)
    results = autocomplete_cso_names(query)
    logging.info("Found %d results", len(results))
    return jsonify(list(results))


@plugin.route("/rm_names/<project_id>", **plugin_options)  # type: ignore[misc]
def rm_names(project_id: str) -> "Response":
    """Autocomplete CSO names.

    Returns:
        The response.
    """
    query: str = "" if not (q := request.args.get("q")) else q
    logging.info("Autocompleting RM names for query `%s`", query)
    results = autocomplete_rm_names(query)
    logging.info("Found %d results", len(results))
    return jsonify(list(results))


@plugin.route(  # type: ignore[misc]
    "/regenerate_company_names/<project_id>", **plugin_options
)
def regenerate_company_names(project_id: str) -> "Response":
    """Regenerate the autocomplete for company names.

    Returns:
        The response.
    """
    global autocomplete_company_names  # noqa: PLW0603
    logging.info("Regenerating company name autocomplete index")
    autocomplete_company_names = regenerate_company_names_autocomplete()
    return jsonify(success=True)


@plugin.route("/regenerate_users/<project_id>", **plugin_options)  # type: ignore[misc]
def regenerate_users(project_id: str) -> "Response":
    """Regenerate the autocomplete for user names.

    Returns:
        The response.
    """
    global autocomplete_cso_names, autocomplete_rm_names  # noqa: PLW0603
    logging.info("Regenerating CSO name autocomplete index")
    autocomplete_cso_names = regenerate_cso_names_autocomplete()
    logging.info("Regenerating RM name autocomplete index")
    autocomplete_rm_names = regenerate_rm_names_autocomplete()
    return jsonify(success=True)
