"""Load the autocomplete indices."""

import json
import logging
from pathlib import Path

from octopus.clients import init_redis_client
from octopus.data import CompanyDataIndex
from octopus.text import Autocomplete

COMPANY_NAMES = "/flash/octopus/cache/autocomplete_company_names.json"
CSO_NAMES = "/flash/octopus/cache/autocomplete_cso_names.json"
RM_NAMES = "/flash/octopus/cache/autocomplete_rm_names.json"


def load_company_names_autocomplete() -> Autocomplete:
    """Load the company names autocomplete index.

    Returns:
        The autocomplete index.
    """
    if not Path(COMPANY_NAMES).exists():
        logging.info("Cache not found, regenerating")
        return regenerate_company_names_autocomplete()
    logging.info("Loading autocomplete for company names from cache")
    return Autocomplete.load_index(COMPANY_NAMES)


def load_cso_names_autocomplete() -> Autocomplete:
    """Load the CSO names autocomplete index.

    Returns:
        The autocomplete index.
    """
    if not Path(CSO_NAMES).exists():
        logging.info("Cache not found, regenerating")
        return regenerate_cso_names_autocomplete()
    logging.info("Loading autocomplete for CSO names from cache")
    return Autocomplete.load_index(CSO_NAMES)


def load_rm_names_autocomplete() -> Autocomplete:
    """Load the RM names autocomplete index.

    Returns:
        The autocomplete index.
    """
    if not Path(RM_NAMES).exists():
        logging.info("Cache not found, regenerating")
        return regenerate_rm_names_autocomplete()
    logging.info("Loading autocomplete for RM names from cache")
    return Autocomplete.load_index(RM_NAMES)


def regenerate_company_names_autocomplete() -> Autocomplete:
    """Regenerate the company names autocomplete index.

    Returns:
        The autocomplete index.
    """
    logging.info("Constructing autocomplete for company names")
    autocomplete = Autocomplete("company_names")
    index = CompanyDataIndex.load_index()
    for company in index._index_cif.values():  # noqa: SLF001
        autocomplete.insert(f"{company['company_name'][0]}|{company['company_cif'][0]}")
    autocomplete.save_index(COMPANY_NAMES)
    return autocomplete


def regenerate_cso_names_autocomplete() -> Autocomplete:
    """Regenerate the CSO names autocomplete index.

    Returns:
        The autocomplete index.
    """
    logging.info("Constructing autocomplete for CSO names")
    autocomplete = Autocomplete("cso_names")
    redis_client = init_redis_client()
    users: dict[str, str] = {
        k.decode(): v.decode()
        for k, v in redis_client.hgetall("user_roles_hash").items()
    }

    for v in users.values():
        user: dict[str, str] = json.loads(v)
        if (name := user.get("name")) and user["role_ocbc"] in {
            "iLMS:BBCACSOMaker",
            "iLMS:BBCACSOChecker",
        }:
            autocomplete.insert(name)
    autocomplete.save_index(CSO_NAMES)
    return autocomplete


def regenerate_rm_names_autocomplete() -> Autocomplete:
    """Regenerate the RM names autocomplete index.

    Returns:
        The autocomplete index.
    """
    logging.info("Constructing autocomplete for RM names")
    autocomplete = Autocomplete("rm_names")
    redis_client = init_redis_client()
    users: dict[str, str] = {
        k.decode(): v.decode()
        for k, v in redis_client.hgetall("user_roles_hash").items()
    }

    for v in users.values():
        user: dict[str, str] = json.loads(v)
        if (name := user.get("name")) and user["role_ocbc"] == "Workbench:RM":
            autocomplete.insert(name)
    autocomplete.save_index(RM_NAMES)
    return autocomplete
