import pytest
from tag_email_metadata import TagEmailMetadata, parse_document_date, parse_email_body


def test_parse_date(valid_date_cases):
    date_string, expected = valid_date_cases
    assert expected == parse_document_date(date_string)


def test_parse_invalid_date(invalid_date_cases):
    with pytest.raises(ValueError):
        parse_document_date(invalid_date_cases)


def test_parse_email_body(email_body: "str"):
    assert parse_email_body(email_body) == {
        "uid": ["ABC", "DEF", "GHI", "JKL", "MNO", "PQR", "STU"],
        "document type": ["report"],
        "document date": ["2022-12-12"],
        "reference": ["R/1234/56789/XYZ"],
        "audit viewing": ["Y"],
    }


class TestTagEmailMetadata:
    def setup_method(self):
        self.tag_email_metadata = TagEmailMetadata()

    def test_empty_email_body(self, item: "dict"):
        assert self.tag_email_metadata.consume(item) == item

    def test_invalid_email_body_type(self, item: "dict"):
        item["keywords"].update({"email_body": "123"})

        assert self.tag_email_metadata.consume(item) == item

    def test_invalid_email_body_value(self, item: "dict"):
        item["keywords"].update({"email_body": ["company uid: ABC"]})

        assert self.tag_email_metadata.consume(item) == item

    def test_add_labels(self, item: "dict", email_body: "str"):
        item["keywords"].update({"email_body": [email_body]})

        assert self.tag_email_metadata.consume(item).get("keywords") == {
            "company_uid": ["ABC", "DEF", "GHI", "JKL", "MNO", "PQR", "STU"],
            "document_type_true": ["report"],
            "document_date_true": ["2022-12-12"],
            "reference_true": ["R/1234/56789/XYZ"],
            "audit_viewing_true": ["Y"],
        }


@pytest.fixture
def item() -> "dict":
    return {
        "id": "987654321",
        "title": "Test item",
        "keywords": {},
    }


@pytest.fixture
def email_body() -> "str":
    return "uid: ABC; DEF;     GHI; JKL ;MNO     ;PQR ;STU;\r\ndocument type: report\r\ndocument date: 2022-12-12\r\nreference: R/1234/56789/XYZ\r\naudit viewing: Y"


@pytest.fixture(
    params=[
        ("02-04-2023", "2023-04-02"),
        ("02-4-2023", "2023-04-02"),
        ("2-04-2023", "2023-04-02"),
        ("2-4-2023", "2023-04-02"),
        ("02/04/2023", "2023-04-02"),
        ("02/4/2023", "2023-04-02"),
        ("2/04/2023", "2023-04-02"),
        ("2/4/2023", "2023-04-02"),
        ("20230402", "2023-04-02"),
        ("2023402", "2023-04-02"),
        ("2023042", "2023-04-02"),
        ("202342", "2023-04-02"),
        ("202304-02", "2023-04-02"),
        ("20234-02", "2023-04-02"),
        ("202304-2", "2023-04-02"),
        ("20234-2", "2023-04-02"),
        ("2023-0402", "2023-04-02"),
        ("2023-402", "2023-04-02"),
        ("2023-042", "2023-04-02"),
        ("2023-42", "2023-04-02"),
        ("2023-04-02", "2023-04-02"),
        ("2023-4-02", "2023-04-02"),
        ("2023-04-2", "2023-04-02"),
        ("2023-4-2", "2023-04-02"),
        ("202304-02", "2023-04-02"),
        ("20234-02", "2023-04-02"),
        ("202304-2", "2023-04-02"),
        ("20234-2", "2023-04-02"),
        ("2023-0402", "2023-04-02"),
        ("2023-402", "2023-04-02"),
        ("2023-042", "2023-04-02"),
        ("2023-42", "2023-04-02"),
        ("2023-04-02", "2023-04-02"),
        ("2023-4-02", "2023-04-02"),
        ("2023-04-2", "2023-04-02"),
        ("2023-4-2", "2023-04-02"),
        ("2023/04/02", "2023-04-02"),
        ("2023/4/02", "2023-04-02"),
        ("2023/04/2", "2023-04-02"),
        ("2023/4/2", "2023-04-02"),
        ("230402", "2023-04-02"),
        ("23402", "2023-04-02"),
        ("23042", "2023-04-02"),
        ("2342", "2023-04-02"),
        ("2304-02", "2023-04-02"),
        ("234-02", "2023-04-02"),
        ("2304-2", "2023-04-02"),
        ("234-2", "2023-04-02"),
        ("23-0402", "2023-04-02"),
        ("23-402", "2023-04-02"),
        ("23-042", "2023-04-02"),
        ("23-42", "2023-04-02"),
        ("23-04-02", "2023-04-02"),
        ("23-4-02", "2023-04-02"),
        ("23-04-2", "2023-04-02"),
        ("23-4-2", "2023-04-02"),
        ("2304-02", "2023-04-02"),
        ("234-02", "2023-04-02"),
        ("2304-2", "2023-04-02"),
        ("234-2", "2023-04-02"),
        ("23-0402", "2023-04-02"),
        ("23-402", "2023-04-02"),
        ("23-042", "2023-04-02"),
        ("23-42", "2023-04-02"),
        ("23-04-02", "2023-04-02"),
        ("23-4-02", "2023-04-02"),
        ("23-04-2", "2023-04-02"),
        ("23-4-2", "2023-04-02"),
        ("23/04/02", "2023-04-02"),
        ("23/4/02", "2023-04-02"),
        ("23/04/2", "2023-04-02"),
        ("23/4/2", "2023-04-02"),
    ],
)
def valid_date_cases(request):
    return request.param


@pytest.fixture(
    params=[
        "",
        "invalid_date_string",
        "2023-01-00",
        "2023-00-01",
        "2023-00-00",
        "2023-01-32",
        "2023-13-01",
        "2023-13-32",
        "202304.20",
        "2023.0420",
        "2023.04.20",
        "2304.20",
        "23.0420",
        "23.04.20",
    ],
)
def invalid_date_cases(request):
    return request.param
