import sys
from datetime import date
from unittest.mock import MagicMock, patch

import pytest

from octopus.clients.wfi_client import WFI_FIELDS

sys.modules["squirro.lib.storage.handler"] = MagicMock()
sys.modules["squirro.service.pdfconversion.controllers"] = MagicMock()

from store_into_wfi import StoreIntoWFI  # noqa: E402


class TestStoreIntoWFI:
    def setup_method(self):
        self.store_into_wfi = StoreIntoWFI({})
        self.store_into_wfi.project_id = "project-id"
        self.store_into_wfi.storage_handler = MagicMock()
        self.store_into_wfi.wfi_client = MagicMock()
        self.store_into_wfi.wfi_client.FIELDS = WFI_FIELDS
        self.store_into_wfi.wfi_client.checkin_document.return_value = (
            "wfi-document-id-1"
        )

    @patch(
        "store_into_wfi.ConvertController._pdf_path",
        return_value="1a/2b/3ccc/wfi-document-id-1.pdf",
    )
    def test_store_into_wfi(self, mock_pdf_path, test_data):
        input_item, expected_output, is_checked_in = test_data
        assert self.store_into_wfi.consume(input_item) == expected_output

        if is_checked_in:
            self.store_into_wfi.wfi_client.checkin_document.assert_called_once()


file_preview = {
    "id": "preview-123",
    "mime_type": "image/png",
    "content_url": "storage://localfile/12/12/121212/test.preview.png",
}


item_invalid = {
    "id": "item-invalid",
    "title": "A Very " + "Long" * 100 + "Title",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {},
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://localfile/12/12/121212/test.pdf",
        },
        file_preview,
    ],
}

item_pdf = {
    "id": "item-pdf",
    "title": "Item PDF",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {},
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://localfile/12/12/121212/test.pdf",
        },
        file_preview,
    ],
}

item_pdf_processed = {
    "id": "item-pdf",
    "title": "Item PDF",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_company_cif": [""],
        "wfi_company_name": [""],
        "wfi_company_segment": ["unassigned"],
        "wfi_company_rm_code": ["NA"],
        "wfi_company_team_name": ["NA"],
        "wfi_document_category": ["Unclassified"],
        "wfi_document_name": ["Unclassified"],
        "wfi_document_type": ["Unclassified"],
        "wfi_document_date": [date.today().strftime("%Y-%m-%d")],
        "wfi_document_id": ["wfi-document-id-1"],
        "wfi_status": ["Success"],
        "wfi_references": [""],
    },
    "files": [
        {
            "id": "wfi-document-id-1",
            "mime_type": "application/pdf",
            "content_url": "storage://wfi/project-id/wfi-document-id-1",
        },
        file_preview,
    ],
}

item_ocr = {
    "id": "item-ocr",
    "title": "Item OCR",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {},
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://localfile/12/12/121212/image.tiff.pdf",
            "pdf_ocr:ocr_status": "success",
        },
        {
            "id": "image-123",
            "mime_type": "image/tiff",
            "content_url": "storage://localfile/12/12/121212/image.tiff",
        },
        file_preview,
    ],
}


item_ocr_processed = {
    "id": "item-ocr",
    "title": "Item OCR",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_company_cif": [""],
        "wfi_company_name": [""],
        "wfi_company_segment": ["unassigned"],
        "wfi_company_rm_code": ["NA"],
        "wfi_company_team_name": ["NA"],
        "wfi_document_category": ["Unclassified"],
        "wfi_document_name": ["Unclassified"],
        "wfi_document_type": ["Unclassified"],
        "wfi_document_date": [date.today().strftime("%Y-%m-%d")],
        "wfi_document_id": ["wfi-document-id-1"],
        "wfi_status": ["Success"],
        "wfi_references": [""],
    },
    "files": [
        {
            "id": "wfi-document-id-1",
            "mime_type": "application/pdf",
            "content_url": "storage://wfi/project-id/wfi-document-id-1",
            "pdf_ocr:ocr_status": "success",
        },
        file_preview,
    ],
}


item_no_ocr = {
    "id": "item-no-ocr",
    "title": "Item No OCR",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {},
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://pdf_conversion/11/11/111111/image.tiff.pdf",
        },
        {
            "id": "image-123",
            "mime_type": "image/tiff",
            "content_url": "storage://localfile/12/12/121212/image.tiff",
        },
        file_preview,
    ],
}

item_no_ocr_processed = {
    "id": "item-no-ocr",
    "title": "Item No OCR",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_company_cif": [""],
        "wfi_company_name": [""],
        "wfi_company_segment": ["unassigned"],
        "wfi_company_rm_code": ["NA"],
        "wfi_company_team_name": ["NA"],
        "wfi_document_category": ["Unclassified"],
        "wfi_document_name": ["Unclassified"],
        "wfi_document_type": ["Unclassified"],
        "wfi_document_date": [date.today().strftime("%Y-%m-%d")],
        "wfi_document_id": ["wfi-document-id-1"],
        "wfi_status": ["Success"],
        "wfi_references": [""],
    },
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://pdf_conversion/1a/2b/3ccc/wfi-document-id-1.pdf",
        },
        {
            "id": "wfi-document-id-1",
            "mime_type": "image/tiff",
            "content_url": "storage://wfi/project-id/wfi-document-id-1",
        },
        file_preview,
    ],
}

item_minor_checkin = {
    "id": "item-minor-checkin",
    "title": "Item Minor Checkin",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_document_id": ["wfi-document-id-0"],
        "source_type": ["WFI Historic"],
    },
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://localfile/12/12/121212/image.tiff.pdf",
            "pdf_ocr:ocr_status": "success",
        },
        {
            "id": "image-123",
            "mime_type": "image/tiff",
            "content_url": "storage://localfile/12/12/121212/image.tiff",
        },
        file_preview,
    ],
}

item_minor_checkin_processed = {
    "id": "item-minor-checkin",
    "title": "Item Minor Checkin",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_document_id": ["wfi-document-id-1"],
        "wfi_status": ["Success"],
        "source_type": ["WFI Historic"],
    },
    "files": [
        {
            "id": "wfi-document-id-1",
            "mime_type": "application/pdf",
            "content_url": "storage://wfi/project-id/wfi-document-id-1",
            "pdf_ocr:ocr_status": "success",
        },
        file_preview,
    ],
}

item_no_checkin = {
    "id": "item-no-checkin",
    "title": "Item No Checkin",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_document_id": ["wfi-document-id-0"],
        "source_type": ["WFI Historic"],
    },
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://pdf_conversion/11/11/111111/image.tiff.pdf",
            "pdf_ocr:ocr_status": "skipped",
        },
        {
            "id": "image-123",
            "mime_type": "image/tiff",
            "content_url": "storage://localfile/12/12/121212/image.tiff",
        },
        file_preview,
    ],
}

item_no_checkin_processed = {
    "id": "item-no-checkin",
    "title": "Item No Checkin",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_document_id": ["wfi-document-id-0"],
        "wfi_status": ["Success"],
        "source_type": ["WFI Historic"],
    },
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://pdf_conversion/1a/2b/3ccc/wfi-document-id-1.pdf",
            "pdf_ocr:ocr_status": "skipped",
        },
        {
            "id": "wfi-document-id-0",
            "mime_type": "image/tiff",
            "content_url": "storage://wfi/project-id/wfi-document-id-0",
        },
        file_preview,
    ],
}

item_unzipped = {
    "id": "item-unzipped",
    "title": "Item Unzipped",
    "created_at": "2024-02-08T10:10:10",
    "wfi:initial_checkin": True,
    "keywords": {},
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://localfile/12/12/121212/test.pdf",
        },
        file_preview,
    ],
}

item_unzipped_processed = {
    "id": "item-unzipped",
    "title": "Item Unzipped",
    "created_at": "2024-02-08T10:10:10",
    "wfi:initial_checkin": True,
    "keywords": {
        "wfi_company_cif": [""],
        "wfi_company_name": [""],
        "wfi_company_segment": ["unassigned"],
        "wfi_company_rm_code": ["NA"],
        "wfi_company_team_name": ["NA"],
        "wfi_document_category": ["Unclassified"],
        "wfi_document_name": ["Unclassified"],
        "wfi_document_type": ["Unclassified"],
        "wfi_document_date": [date.today().strftime("%Y-%m-%d")],
        "wfi_document_id": ["wfi-document-id-1"],
        "wfi_status": ["Success"],
        "wfi_references": [""],
    },
    "files": [
        {
            "id": "wfi-document-id-1",
            "mime_type": "application/pdf",
            "content_url": "storage://wfi/project-id/wfi-document-id-1",
        },
        file_preview,
    ],
}


item_with_keywords = {
    "id": "item-with-keywords",
    "title": "Item With Keywords",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_company_cif": ["11111"],
        "wfi_company_name": ["Company 1"],
        "wfi_company_segment": ["R"],
        "wfi_company_rm_code": ["R001"],
        "wfi_company_team_name": ["Team 1"],
        "wfi_document_category": ["Document Category"],
        "wfi_document_name": ["Document Name"],
        "wfi_document_type": ["Document TYpe"],
        "wfi_document_date": [date.today().strftime("%Y-%m-%d")],
    },
    "files": [
        {
            "id": "file-123",
            "mime_type": "application/pdf",
            "content_url": "storage://localfile/12/12/121212/test.pdf",
        },
        file_preview,
    ],
}

item_with_keywords_processed = {
    "id": "item-with-keywords",
    "title": "Item With Keywords",
    "created_at": "2024-02-08T10:10:10",
    "keywords": {
        "wfi_company_cif": ["11111"],
        "wfi_company_name": ["Company 1"],
        "wfi_company_segment": ["R"],
        "wfi_company_rm_code": ["R001"],
        "wfi_company_team_name": ["Team 1"],
        "wfi_document_category": ["Document Category"],
        "wfi_document_name": ["Document Name"],
        "wfi_document_type": ["Document TYpe"],
        "wfi_document_date": [date.today().strftime("%Y-%m-%d")],
        "wfi_document_id": ["wfi-document-id-1"],
        "wfi_status": ["Success"],
        "wfi_references": [""],
    },
    "files": [
        {
            "id": "wfi-document-id-1",
            "mime_type": "application/pdf",
            "content_url": "storage://wfi/project-id/wfi-document-id-1",
        },
        file_preview,
    ],
}


@pytest.fixture(
    params=[
        (item_invalid, None, False),
        (item_pdf, item_pdf_processed, True),
        (item_ocr, item_ocr_processed, True),
        (item_no_ocr, item_no_ocr_processed, True),
        (item_minor_checkin, item_minor_checkin_processed, True),
        (item_no_checkin, item_no_checkin_processed, False),
        (item_unzipped, item_unzipped_processed, True),
        (item_with_keywords, item_with_keywords_processed, True),
    ],
)
def test_data(request):
    return request.param
